#' NAP_prior: Derive NAP/mNAP/eNAP priors
#' @description
#' Builds the informative NAP component (mean/variance from the indirect path)
#' and the vague component, and reports the mixing weight depending on the mode:
#' - \code{weight_mtd = "fixed"}: use the supplied fixed weight \code{w in [0,1]}.
#' - \code{weight_mtd = "adaptive"} (eNAP): if \code{y_EC2} is provided, compute the
#'   data-dependent weight via the elastic link; otherwise, print a formula note.
#'
#' Derive NAP-based prior (s) based on indirect evidence
#' 
#' @description
#' Derive the NAP-based posteriors with provided summary statistics on indirect evidence edges
#' By default, the function assumes a vague component is desired, as a result, to obtain NAP/mNAP/eNAP:
#' \itemize{
#'   \item \emph{NAP} Set \code{weight_mtd="fixed"} and \code{w=1}, use the NAP (informative component) column results
#'   \item \emph{mNAP} Set \code{weight_mtd="fixed"} and  \code{w} as pre-specified fixed weight. The resulting mNAP is \eqn{w\pi_{NAP}+(1-w)\pi_0}
#'   \item \emph{eNAP} Set \code{weight_mtd="adaptive"} and provide calibrated \code{a} and \code{b} as from tune_param_eNAP function, then either:
#'   1). Provide assumed value for \code{y_EC2} and \code{s_EC2} (i.e., as for sample size calculation): return a calculated dynamic weight \eqn{w(Z)}, the resulting 
#'      eNAP is then \eqn{w(Z)\pi_{NAP}+(1-w(Z))\pi_0};
#'   OR
#'   2). Leave  \code{y_EC2} and \code{s_EC2} as NULL, return the NAP (informative component) and Vague component, with description for protocol reference
#' }
#' @details
#' This function automatically selects one external trial vs multiple external trials setting:
#' - One external trial if \code{length(y_C2C1) == 1 & length(s_C2C1)==1}  (one external trial).
#' - Multiple external trials if \code{length(y_C2C1) > 1 & length(s_C2C1)==length(y_C2C1)}. By default uses \code{metafor::rma.uni(..., method="REML")} to obtain REML estimate;
#'   Otherwise please provide \code{sigma2_hat} 
#' @return Displays the NAP prior as a mixture of an informative prior (constructed based on 
#' the indirect evidence path) and a vague prior. 
#'
#' @examples
#' ## ------------------------------------------------------------
#' ## Example 1: One external trial setting with fixed mixing weight of 0.5 (mNAP)
#' ## ------------------------------------------------------------
#' mNAP_test1 <- NAP_prior(
#'   weight_mtd = "fixed", w = 0.50,                  # fixed mixture weight
#'   y_EC1  = -0.36, s_EC1  = 0.16^2,
#'   y_C2C1 = -0.30, s_C2C1 = 0.14^2,                  # single external trial
#'   tau0   = 1000
#' )
#' print(mNAP_test1)  
#' plot(mNAP_test1)   
#'
#' ## ------------------------------------------------------------
#' ## Example 2: RE case (multiple historical), ADAPTIVE weight
#' ## ------------------------------------------------------------
#' eNAP_test1 <- NAP_prior(
#'   weight_mtd = "adaptive",
#'   a = -2, b = 10,            # from calibration
#'   y_EC1  = -0.36, s_EC1  = 0.16^2,                 # E:C1 (current, pre-change)
#'   y_C2C1 = c(-0.28, -0.35, -0.31),                 # C2:C1 (external trials)
#'   s_C2C1 = c(0.12^2, 0.11^2, 0.15^2),
#'   tau0   = 1000                                     # vague variance
#' )
#' print(eNAP_test1)
#'
#'    
#' @param weight_mtd Either `"adaptive"` (eNAP) or `"fixed"` (NP/NAP/mNAP).
#' @param w Fixed prior weight in \[0,1\]; **required only** if `weight_mtd="fixed"`. Ignored otherwise.
#'   `0<w<1` infers mixture NAP; `w=0` infers NP; `w=1` infers NAP.
#' @param a,b eNAP tuning parameters; **required only** if `weight_mtd="adaptive"`
#'   (`a<0` and `b>0`). Ignored in fixed mode.
#' @param y_EC2,s_EC2 Log-HR and SE for \eqn{E:C2} (Current trial post-SoC change).
#' @param y_EC1,s_EC1 Log-HR and SE for \eqn{E:C1} (Current trial pre-SoC change).
#' @param y_C2C1,s_C2C1 Historical C2 vs. C1 trial Log-HRs and SEs. 
#' @param sigma2_hat Positive scalar, required only for multiple external trials setting, leave blank if use default REML estimate, otherwise provide user-specified value
#' @param mu0,tau0 mean and variance of the vague component (default sqrt(1000)).
#' @param lambda Randomization ratio (default 1).
#' @return An object of class "NAPrior" (data.frame + attributes).
#' @importFrom stats dnorm
#' @importFrom graphics mtext par abline lines plot.new text
#' @importFrom stats rexp rweibull
#' @export
#' 
NAP_prior <- function(weight_mtd = c("adaptive","fixed"),
                       w   = NULL,           # for fixed [0,1]
                       a   = NULL, b = NULL, # for adaptive (b>0)
                       # direct data (needed only if adaptive):
                       y_EC2  = NULL, s_EC2 = NULL,
                       # indirect path (always needed):
                       y_EC1,  s_EC1,
                       y_C2C1, s_C2C1,
                       mu0=0,  tau0 = 1000,  
                       lambda = 1,
                       sigma2_hat = NULL 
                       ) {
  weight_mtd <- match.arg(weight_mtd)
  req_num <- function(x, nm) if (!is.numeric(x) || length(x)!=1L || !is.finite(x)) stop("`", nm, "` must be a finite scalar.")
  req_pos <- function(x, nm) { req_num(x, nm); if (x <= 0) stop("`", nm, "` must be > 0.") }
  
  req_num(y_EC1, "y_EC1");  req_pos(s_EC1, "s_EC1")
  if (!is.numeric(y_C2C1) || !is.numeric(s_C2C1))
    stop("`y_C2C1` and `s_C2C1` must be numeric.", call. = FALSE)
  if (length(y_C2C1) != length(s_C2C1) || length(y_C2C1) < 1L)
    stop("`y_C2C1` and `s_C2C1` must have the same positive length.", call. = FALSE)
  if (any(!is.finite(y_C2C1)) || any(!is.finite(s_C2C1)) || any(s_C2C1 <= 0))
    stop("Non-finite or non-positive values in `y_C2C1`/`s_C2C1`.", call. = FALSE)
  req_pos(tau0, "tau0")
  
  
  if (weight_mtd == "fixed") {
    if (is.null(w) || !is.finite(w) || w < 0 || w > 1)
      stop("For mNAP, please provide fixed weight `w` in [0,1].", call. = FALSE)
  } else { # adaptive
    if (is.null(a) || is.null(b) || !is.finite(a) || !is.finite(b) || !(a < 0) || !(b > 0))
      stop("For eNAP, please provide elastic parameters with `a<0` and `b>0`.", call. = FALSE)
  }
  original_s_C2C1=s_C2C1
  original_y_C2C1=y_C2C1
  original_sigma2_hat=sigma2_hat
  ## --- Mode selection (one/multiple external trial) ---
  mode <- if (length(s_C2C1) == 1L) "One external trial" else "Multiple external trials"
  if (mode == "Multiple external trials") {
    if (is.null(sigma2_hat)){
      reml <- metafor::rma.uni(yi = y_C2C1, vi = s_C2C1, method = "REML")
      sigma2_hat<-reml$tau2
    } 
    if (!is.finite(sigma2_hat) || sigma2_hat < 0) stop("`sigma2_hat` must be >= 0.", call. = FALSE)
    s_C2C1  <- 1 / sum(1 / (s_C2C1 + sigma2_hat))
    y_C2C1 <- as.numeric(reml$beta)  
  }
  
  
  v_1 <- 1/(1/s_EC1+1/tau0)
  v_2 <- 1/(1/s_C2C1+1/tau0)
  V_ind <- v_1 + v_2
  
  mu_ind<-v_1*(mu0/tau0+y_EC1/s_EC1)-v_2*(mu0/tau0+y_C2C1/s_C2C1)

  # --- evaluate weight
  note <- NULL
  if (weight_mtd == "fixed") {
    w_inform <- w
  } else {
    # adaptive (eNAP)
    if (!is.null(y_EC2) && !is.null(s_EC2)) {
      req_num(y_EC2, "y_EC2"); req_pos(s_EC2, "s_EC2")
      # Bucher x = y_EC2 - (y_EC1 - y_C2C1_pool)
      Z  <- abs(y_EC2 - (y_EC1 - y_C2C1))/sqrt(sum(s_EC2,s_EC1,s_C2C1))
      # Denominator: FE uses s_C2C1; RE uses v2 (the posterior variance piece for d)
      w_inform <- 1 / (1 + exp(a + b * log(Z + 1)))
    } else {
      w_inform <- NA_real_
      note <- paste0(
        "Dynamic weight not available yet; eNAP uses data-dependent weight evaluated as w(Z) = 1/(1+exp(a + b*log(Z+1))),  ",
        "where Bucher test statistic Z = |y_EC2 - (y_EC1 - y_C2C1)| / sqrt(s_EC2 + s_EC1 + s_C2C1). ",
        "Parameters (a,b) are tuning parameters that shall be calibrated by tune_param_eNAP function",
        "See Zhang et al., NAP manuscript, for details."
      )
    }
  }
  w_vague <- if (is.na(w_inform)) NA_real_ else (1 - w_inform)

  ess_factor <- (lambda + 1/lambda + 2)
  ESS_informative <- ess_factor / V_ind
  
  ESS_vague <- NA_real_
  
  # --- assemble table
  tab <- data.frame(
    `NAP (Informative)` = c(ifelse(is.na(w_inform),"TBD",w_inform), mu_ind, V_ind,ESS_informative),
    Vague       = c(ifelse(is.na(w_vague),"TBD",w_vague),  mu0,    tau0,ESS_vague),
    row.names   = c("Mixing Weight", "Mean", "Variance","ESS (events)"),
    check.names = FALSE
  )
  
  # store and return
x <- list(
  table  = tab,                       # your 2-column table (Informative/Vague)
  mode   = mode,
  weight_mode = weight_mtd,           # "fixed" or "adaptive"
  details = list(
    a = a, b = b,                     # may be NA for fixed mode
    lambda = lambda,
    y_inputs=list(
      y_EC2 = y_EC2,
      y_EC1 = y_EC1,
      y_C2C1_original = original_y_C2C1,
      y_C2C1_RE = if (length(original_y_C2C1) > 1L) y_C2C1 else NA_real_  # your RE v2
    ),
    # keep the exact variances used to build Z:
    s_inputs = list(
      s_EC2 = s_EC2,                  # variance
      s_EC1 = s_EC1,                  # variance
      s_C2C1_original = original_s_C2C1,
      s_C2C1_RE = if (length(original_s_C2C1) > 1L) s_C2C1 else NA_real_ # your RE v2
    ),
    w_inform=w_inform,
    mu_ind = mu_ind, 
    V_ind = V_ind,
    mu0=mu0,
    tau0=tau0,
    ESS_informative=ESS_informative
  ),
  note = note
  )
  class(x) <- "NAP_prior"
  return(x)
}
#' @export
plot.NAP_prior <- function(x,
                           xlim = NULL,
                           n = 1001,
                           lwd = 2,
                           col_mix = "black",
                           main = NULL,
                           ...) {
  stopifnot(inherits(x, "NAP_prior"))
  
  det    <- x$details
  mu_inf <- det$mu_ind
  v_inf  <- det$V_ind
  mu_vag <- det$mu0
  v_vag  <- det$tau0
  
  w_num  <- det$w_inform
  
  ## If dynamic eNAP with no assumed direct data: no well-defined mixture weight
  if (is.na(w_num) || !is.finite(w_num)) {
    message("Dynamic weight not available for adaptive eNAP prior without assumed direct data; cannot plot mixture density.")
    
    op <- par(no.readonly = TRUE)
    on.exit(par(op), add = TRUE)
    plot.new()
    text(
      0.5, 0.6,
      "Dynamic weight not available\nfor adaptive eNAP prior\n(without assumed direct data).",
      cex = 0.9
    )
    return(invisible(NULL))
  }
  
  
  if (!is.finite(mu_inf) || !is.finite(v_inf) || v_inf <= 0)
    stop("Invalid informative component.")
  if (!is.finite(mu_vag) || !is.finite(v_vag) || v_vag <= 0)
    stop("Invalid vague component.")
  
  sd_inf <- sqrt(v_inf)
  sd_vag <- sqrt(v_vag)
  sd_ref <- max(sd_inf, sd_vag)
  
  ## 1) Rough grid to locate the mode of the mixture
  # Use a broad range covering both components
  rough_xlim <- c(
    min(mu_inf - 2.5 * sd_inf, mu_vag - 2.5 * sd_inf),
    max(mu_inf + 2.5 * sd_inf, mu_vag + 2.5 * sd_inf)
  )
  rough_grid <- seq(rough_xlim[1], rough_xlim[2], length.out = 501L)
  d_inf_rough <- stats::dnorm(rough_grid, mean = mu_inf, sd = sd_inf)
  d_vag_rough <- stats::dnorm(rough_grid, mean = mu_vag, sd = sd_vag)
  d_mix_rough <- w_num * d_inf_rough + (1 - w_num) * d_vag_rough
  
  mode_x <- rough_grid[which.max(d_mix_rough)]
  
  ## 2) Final x-range centered at mode
  if (is.null(xlim)) {
    width <- 4 * sd_inf
    xlim  <- mode_x + c(-width, width)
    half  <- diff(xlim) / 2
    if (half > 10) {
      # cap overly wide range to +/-10 around mode
      xlim <- mode_x + c(-10, 10)
    }
  }
  
  ## 3) Final grid and mixture density
  grid  <- seq(xlim[1], xlim[2], length.out = max(201L, as.integer(n)))
  d_inf <- stats::dnorm(grid, mean = mu_inf, sd = sd_inf)
  d_vag <- stats::dnorm(grid, mean = mu_vag, sd = sd_vag)
  d_mix <- w_num * d_inf + (1 - w_num) * d_vag
  
  y_max <- 1.05 * max(d_mix)
  
  if (is.null(main)) {
    main <- sprintf("NAP prior")
  }
  
  op <- par(no.readonly = TRUE)
  on.exit(par(op), add = TRUE)
  
  plot(
    grid, d_mix,
    type = "l",
    lwd  = lwd,
    col  = col_mix,
    xlab = expression(theta[E*C[2]]),
    ylab = "Density",
    main = main,
    xlim = xlim,
    ylim = c(0, y_max),
    ...
  )
  
  invisible(list(
    x      = grid,
    d_mix  = d_mix,
    w      = w_num,
    mode_x = mode_x
  ))
}


