# LogisModExp.R
#' Logistic Modified Exponential Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Logistic Modified Exponential distribution.
#'
#' The Logistic Modified Exponential distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Logistic Modified Exponential distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \beta, \lambda) =
#' \quad 1-\frac{1}{1+\left[\exp \left\{\lambda x 
#' e^{\beta x}\right\}-1\right]^\alpha}   \, ; x \geq 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dlogis.mod.exp()} — Density function
#'   \item \code{plogis.mod.exp()} — Distribution function
#'   \item \code{qlogis.mod.exp()} — Quantile function
#'   \item \code{rlogis.mod.exp()} — Random generation
#'   \item \code{hlogis.mod.exp()} — Hazard function
#' }
#'
#' @usage
#' dlogis.mod.exp(x, alpha, beta, lambda, log = FALSE)
#' plogis.mod.exp(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qlogis.mod.exp(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rlogis.mod.exp(n, alpha, beta, lambda)
#' hlogis.mod.exp(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dlogis.mod.exp}: numeric vector of (log-)densities
#'   \item \code{plogis.mod.exp}: numeric vector of probabilities
#'   \item \code{qlogis.mod.exp}: numeric vector of quantiles
#'   \item \code{rlogis.mod.exp}: numeric vector of random variates
#'   \item \code{hlogis.mod.exp}: numeric vector of hazard values
#' }
#'
#' @references  
#' Chaudhary, A.K., & Kumar, V.(2020).
#' A Study on Properties and Applications of Logistic Modified Exponential
#' Distribution. \emph{International Journal of Latest Trends In Engineering 
#' and Technology (IJLTET)},\bold{18(1)},19--29.     
#'  
#' @examples
#' x <- seq(0.1, 2.0, 0.2)
#' dlogis.mod.exp(x, 1.5, 1.5, 0.2)
#' plogis.mod.exp(x, 1.5, 1.5, 0.2)
#' qlogis.mod.exp(0.5, 1.5, 1.5, 0.2)
#' rlogis.mod.exp(10, 1.5, 1.5, 0.2)
#' hlogis.mod.exp(x, 1.5, 1.5, 0.2)
#' 
#' # Data
#' x <- stress
#' # ML estimates
#' params = list(alpha=2.0354, beta=0.1891, lambda=0.1656)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = plogis.mod.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qlogis.mod.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dlogis.mod.exp, pfun=plogis.mod.exp, plot=TRUE)
#' print.gofic(out) 
#'
#' @name LogisModExp
#' @aliases dlogis.mod.exp plogis.mod.exp qlogis.mod.exp rlogis.mod.exp hlogis.mod.exp
#' @export
NULL

#' @rdname LogisModExp
#' @usage NULL
#' @export   
dlogis.mod.exp <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
	z <- x[valid] * beta
    u <- exp(lambda * x[valid] * exp(z))
    num <- alpha * lambda * exp(z) * (1+z) * u *  ((u-1)^(alpha - 1.0))
    deno <- (1.0 + (u-1) ^ alpha) ^ 2.0
    pdf[valid] <- num/deno
	}
    if (log) 
       pdf<- log(pdf)
    return(pdf)   
}

#' @rdname LogisModExp
#' @usage NULL
#' @export  
plogis.mod.exp <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
    u <- exp(lambda * q[valid] * exp(q[valid] * beta))
	tmp <- 1.0 + ((u - 1.0) ^ alpha)
    cdf[valid] <- 1.0 - (1.0/tmp)
	}
    if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname LogisModExp
#' @usage NULL
#' @export  
qlogis.mod.exp <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Parameters 'alpha', 'beta', and 'lambda' must be positive.")

	if (log.p) p <- exp(p)
	if (!lower.tail) p <- 1 - p

	if (any(p < 0 | p > 1))
    stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	is_zero <- p == 0
	is_one  <- p == 1
	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
 
	to_solve <- which(!(is_zero | is_one))
	if (length(to_solve) > 0) {

	root_fun <- function(x, wi) {
	    log(x)+(beta*x)-log((1/lambda) * log(1.0 + ((wi/(1.0 - wi)) ^ (1.0/alpha))))   
	}
	for (i in to_solve) {
      wi <- p[i]
      lower <- .Machine$double.eps * 10   
      upper <- 1e4
      f_low <- root_fun(lower, wi)
      f_up  <- root_fun(upper, wi)

      expand_counter <- 0
      
	while (is.na(f_low) || is.na(f_up) || sign(f_low) == sign(f_up)) {
        upper <- upper * 10
        f_up <- root_fun(upper, wi)
        expand_counter <- expand_counter + 1
        if (expand_counter > 10) break
      }

      sol  <-  tryCatch(
        uniroot(function(x) root_fun(x, wi), c(lower, upper), tol = 1e-8),
        error = function(e) {
          stop(sprintf("Failed to find quantile for p = %g. Try different bracket or check parameters.", wi))
        }
      )
      qtl[i] <- sol$root
    }
  }

  return(qtl)
} 

#' @rdname LogisModExp
#' @usage NULL
#' @export   
rlogis.mod.exp <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qlogis.mod.exp(u, alpha, beta, lambda)
    return(q)
}

#' @rdname LogisModExp
#' @usage NULL
#' @export 
hlogis.mod.exp <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
  
    nume <- dlogis.mod.exp(x, alpha, beta, lambda)
	surv <- 1.0 - plogis.mod.exp(x, alpha, beta, lambda)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    

    return(hrf)    
} 


