% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hsaft.R
\name{hsaft}
\alias{hsaft}
\title{Function to implement the horseshoe shrinkage prior in Bayesian survival regression}
\usage{
hsaft(ct, X, method.tau = c("fixed", "truncatedCauchy", "halfCauchy"),
  tau = 1, method.sigma = c("fixed", "Jeffreys"), Sigma2 = 1,
  burn = 1000, nmc = 5000, thin = 1, alpha = 0.05)
}
\arguments{
\item{ct}{Response, a \eqn{n*2} matrix with first column as response and second column as right censored indicator,
1 is event time and 0 is right censored.}

\item{X}{Matrix of covariates, dimension \eqn{n*p}.}

\item{method.tau}{Method for handling \eqn{\tau}. Select "truncatedCauchy" for full
Bayes with the Cauchy prior truncated to [1/p, 1], "halfCauchy" for full Bayes with
the half-Cauchy prior, or "fixed" to use a fixed value (an empirical Bayes estimate,
for example).}

\item{tau}{Use this argument to pass the (estimated) value of \eqn{\tau} in case "fixed"
is selected for method.tau. Not necessary when method.tau is equal to"halfCauchy" or
"truncatedCauchy". The default (tau = 1) is not suitable for most purposes and should be replaced.}

\item{method.sigma}{Select "Jeffreys" for full Bayes with Jeffrey's prior on the error
variance \eqn{\sigma^2}, or "fixed" to use a fixed value (an empirical Bayes
estimate, for example).}

\item{Sigma2}{A fixed value for the error variance \eqn{\sigma^2}. Not necessary
when method.sigma is equal to "Jeffreys". Use this argument to pass the (estimated)
value of Sigma2 in case "fixed" is selected for method.sigma. The default (Sigma2 = 1)
is not suitable for most purposes and should be replaced.}

\item{burn}{Number of burn-in MCMC samples. Default is 1000.}

\item{nmc}{Number of posterior draws to be saved. Default is 5000.}

\item{thin}{Thinning parameter of the chain. Default is 1 (no thinning).}

\item{alpha}{Level for the credible intervals. For example, alpha = 0.05 results in
95\% credible intervals.}
}
\value{
\item{SurvivalHat}{Predictive survival probability.}
\item{LogTimeHat}{Predictive log time.}
\item{BetaHat}{Posterior mean of Beta, a \eqn{p} by 1 vector.}
\item{LeftCI}{The left bounds of the credible intervals.}
\item{RightCI}{The right bounds of the credible intervals.}
\item{BetaMedian}{Posterior median of Beta, a \eqn{p} by 1 vector.}
\item{Sigma2Hat}{Posterior mean of error variance \eqn{\sigma^2}. If method.sigma =
"fixed" is used, this value will be equal to the user-selected value of Sigma2
passed to the function.}
\item{TauHat}{Posterior mean of global scale parameter tau, a positive scalar.
If method.tau = "fixed" is used, this value will be equal to the user-selected value
of tau passed to the function.}
\item{BetaSamples}{Posterior samples of Beta.}
\item{TauSamples}{Posterior samples of tau.}
\item{Sigma2Samples}{Posterior samples of Sigma2.}
\item{LikelihoodSamples}{Posterior Samples of likelihood.}
}
\description{
This function employs the algorithm provided by van der Pas et. al. (2016) for
log normal Accelerated Failure Rate (AFT) model to fit survival regression. The censored observations are updated
according to the data augmentation of approach of Tanner and Wong (1984).
}
\details{
The model is:
 \eqn{t_i} is response,
 \eqn{c_i} is censored time,
 \eqn{t_i^* = \min_(t_i, c_i)} is observed time,
 \eqn{w_i} is censored data, so \eqn{w_i = \log t_i^*} if \eqn{t_i} is event time and
 \eqn{w_i = \log t_i^*} if \eqn{t_i} is right censored
 \eqn{\log t_i=X\beta+\epsilon, \epsilon \sim N(0,\sigma^2)}
}
\examples{
burnin <- 500   # number of burnin
nmc    <- 1000  # number of Markov Chain samples
y.sd   <- 1     # standard deviation of the data
p      <- 80    # number of covariates
n      <- 40    # number of samples
beta   <- as.vector(smoothmest::rdoublex(p))  # from double exponential distribution
x      <- mvtnorm::rmvnorm(n, mean = rep(0, p))  # from multivariate normal distribution
y.mu   <- x \%*\% beta  # mean of the data
y      <- as.numeric(stats::rnorm(n, mean = y.mu, sd = y.sd))  # from normal distribution
T      <- exp(y)   # AFT model
C      <- rgamma(n, shape = 1.75, scale = 3)  # censoring time
time   <- pmin(T, C)  # observed time is min of censored and true
status = time == T   # set to 1 if event is observed
ct     <- as.matrix(cbind(time = time, status = status))  # censored time

posterior.fit <- hsaft(ct, x, method.tau = "truncatedCauchy", method.sigma = "Jeffreys",
                       burn = burnin, nmc = nmc)
summary(posterior.fit$BetaHat)


}
\references{
Stephanie van der Pas, James Scott, Antik Chakraborty and Anirban Bhattacharya (2016). horseshoe:
Implementation of the Horseshoe Prior. R package version 0.1.0.
https://CRAN.R-project.org/package=horseshoe

Arnab Kumar Maity, Anirban Bhattacharya, Bani K. Mallick, and Veerabhadran Baladandayuthapani (2017).
Joint Bayesian Estimation and Variable Selection for TCPA Protein Expression Data
}
