#' Calculate ddCt Statistics
#'
#' Performs the complete Delta-Delta Ct method with error propagation.
#'
#' @param data Output from import_plate().
#' @param ref_gene Name of the housekeeping gene (e.g., "GAPDH").
#' @param control_group Name of the control condition (e.g., "WT").
#' @return A comprehensive dataframe with Fold Changes, Log2FC, P-values, and Error bars.
#' @examples
#' # Mock data: 2 biological replicates per group to allow t-test
#' df <- data.frame(
#'   Sample = c("S1", "S2", "S3", "S4"),
#'   Group = c("WT", "WT", "Treated", "Treated"),
#'   Gene = c(rep("GAPDH", 4), rep("Target", 4)),
#'   Ct = c(20, 20.1, 20.2, 20.3,   # GAPDH (Consistent)
#'          25, 24.8, 21, 21.2)     # Target (Down in Treated)
#' )
#'
#' calculate_ddct(df, ref_gene = "GAPDH", control_group = "WT")
#' @export
calculate_ddct <- function(data, ref_gene, control_group) {

  # A. Technical Replicate Stats
  tech_stats <- data %>%
    dplyr::group_by(Sample, Gene, Group) %>%
    dplyr::summarise(
      mean_ct = mean(Ct, na.rm = TRUE),
      se_ct = sd(Ct, na.rm = TRUE) / sqrt(dplyr::n()),
      .groups = "drop"
    )

  # B. Delta Ct (Target - Reference)
  ref_data <- tech_stats %>%
    dplyr::filter(Gene == ref_gene) %>%
    dplyr::select(Sample, ref_mean = mean_ct, ref_se = se_ct)

  delta_stats <- tech_stats %>%
    dplyr::filter(Gene != ref_gene) %>%
    dplyr::left_join(ref_data, by = "Sample") %>%
    dplyr::mutate(
      delta_ct = mean_ct - ref_mean,
      se_delta_ct = sqrt(se_ct^2 + ref_se^2) # Error Propagation
    )

  # C. Delta Delta Ct
  control_baseline <- delta_stats %>%
    dplyr::filter(Group == control_group) %>%
    dplyr::group_by(Gene) %>%
    dplyr::summarise(control_mean_dct = mean(delta_ct, na.rm = TRUE))

  final_results <- delta_stats %>%
    dplyr::left_join(control_baseline, by = "Gene") %>%
    dplyr::mutate(
      dd_ct = delta_ct - control_mean_dct,
      fold_change = 2^(-dd_ct),
      log2_fc = -dd_ct,
      fc_min = 2^(-(dd_ct + se_delta_ct)),
      fc_max = 2^(-(dd_ct - se_delta_ct))
    )

  # D. Stats (Compare Specific Group vs Control Group)

  # 1. Isolate Control Values per Gene
  ctrl_values <- delta_stats %>%
    dplyr::filter(Group == control_group) %>%
    dplyr::group_by(Gene) %>%
    dplyr::summarise(ctrl_vec = list(delta_ct), .groups = "drop")

  # 2. Compare Every Group vs Control Vector
  p_values_df <- delta_stats %>%
    dplyr::group_by(Gene, Group) %>%
    dplyr::summarise(test_vec = list(delta_ct), .groups = "drop") %>%
    dplyr::left_join(ctrl_values, by = "Gene") %>%
    dplyr::rowwise() %>%
    dplyr::mutate(
      p_val = tryCatch({
        # Only run test if it's NOT the control group and we have enough data
        if(Group == control_group) {
          NA_real_
        } else {
          t.test(unlist(test_vec), unlist(ctrl_vec))$p.value
        }
      }, error = function(e) NA_real_)
    ) %>%
    dplyr::select(Gene, Group, p_val)

  # Merge stats back to main results
  final_results <- final_results %>%
    dplyr::left_join(p_values_df, by = c("Gene", "Group"))

  return(final_results)
}
