#' Find peaks in a data set
#'
#' \code{find_peaks} finds the peaks, and troughs if requested, of a data set.
#'
#' The filter is implemented by comparing pairs of peaks
#' to a hyperbola determined by the y-intercept (\code{minYerror}),
#' and the slope (\code{minSlope}). The user can employ either an
#' edge filter or a region of interest (\code{ROI}) to filter out
#' certain regions. The difference is that the edge
#' filter waits for the data to be processed, then
#' removes the peaks within a certain distance of the edges.
#' The region of interest lets the user specify a region of the
#' data, and the function removes everything outside that region
#' before doing any analysis.
#'
#' @param dataSet The data to search
#' @param xField The name of the field to use as the x-value
#' @param yField The name of the field to use as the y-value
#' @param minYerror The minimum vertical separation between
#' adjacent peaks and troughs. If the difference is
#' less than this, they are filtered out as noise.
#' If \code{asFraction} is true, this is multiplied by the
#' total y-axis range (y-range) of the data.
#' @param minSlope The minimum slope between adjacent peaks
#' and troughs. Useful when adjacent peaks and troughs
#' are far apart along the x-axis. If \code{asFraction}
#' is true, this is multiplied by the y-range/x-range
#' of the data.
#' @param maxPeakWidth The maximum width a peak can have.
#' Useful for preventing the function from interpreting
#' a long, flat plateau as a peak.
#' @param asFraction Whether to interpret the \code{minYerror},
#' \code{minSlope}, \code{edgeFilter}, and \code{maxPeakWidth}
#' arguments as fractions of the total dimensions
#' of the data (\code{TRUE}) or just as absolute values
#' (\code{FALSE})
#' @param globalFilter Whether to apply the global filter
#' (\code{TRUE}) or just compare peaks and troughs to
#' adjacent peaks and troughs (\code{FALSE})
#' @param ROI Region of interest. Specifies the range of
#' x-values that the function sees. Use the format
#' \code{ROI = c(min, max)}. If this parameter is used,
#' the \code{edgeFilter} parameter is ignored.
#' @param edgeFilter Removes peaks and troughs within this
#' distance of the edge of the data set. If
#' \code{asFraction} is true, this is multiplied by the
#' x-range of the data. This filter is only applied after
#' the data has been processed.
#' @param justPeaks Whether to return only the peaks (\code{TRUE})
#' or also the troughs (\code{FALSE})
#'
#' @return A data frame containing the peaks of the data.
#' @import dplyr
#' @importFrom magrittr '%>%'
#' @importFrom magrittr 'or'
#' @importFrom stats 'complete.cases'
#' @importFrom rlang '.data'
#' @export
find_peaks <- function(dataSet,
                       xField,
                       yField,
                       minYerror = 0.1,
                       minSlope = 0,
                       asFraction = TRUE,
                       maxPeakWidth = Inf,
                       globalFilter = TRUE,
                       ROI = c(NA, NA),
                       edgeFilter = 0.02,
                       justPeaks = TRUE) {

  if (!all(xField %in% names(dataSet) &
           yField %in% names(dataSet))) {
    message('Please check your xField and yField inputs.')
    return(NULL)
  }

  df <- select(dataSet, X=!!xField, Y=!!yField) %>%
    filter(complete.cases(.))

  if (any(is.na(ROI))){
    xMax <- max(df$X)
    xMin <- min(df$X)
  } else {
    xMax <- ROI[2]
    xMin <- ROI[1]
    df <- filter(df, .data$X >= xMin, .data$X <= xMax)
    edgeFilter <- 0
  }

  if (asFraction) {
    yRange <- max(df$Y) - min(df$Y)
    xRange <- xMax - xMin
    slopeRange <- yRange / xRange
    minYerror <- minYerror * yRange
    minSlope <- minSlope * slopeRange
    edgeFilter <- edgeFilter * xRange
    maxPeakWidth <- maxPeakWidth * xRange
  }

  # Collect only the necessary fields from the input data frame
  PnT <- df %>%
    # Ensure the x field is ascending
    arrange(.data$X) %>%
    # Determine whether the values are increasing or decreasing,
    # then identify points as peaks or troughs
    mutate(Increasing = case_when(.data$Y < lead(.data$Y) ~ 1,
                                  .data$Y > lead(.data$Y) ~ 0,
                                  .data$Y == lead(.data$Y) ~ 2),
           Type = ifelse((.data$Increasing==0) | (lag(.data$Increasing)==1),
                         'Peak', 'Trough')) %>%
    filter(.data$Increasing != lag(.data$Increasing), complete.cases(.)) %>%
    select(-.data$Increasing)

  # If the user naively specified zero filtering conditions
  # give them all the peaks and troughs
  if (minYerror == 0 & minSlope == 0 & maxPeakWidth > xMax - xMin) {
    PnT <- mutate(PnT, Type=as.factor(.data$Type))
    if (justPeaks) {
      PnT <- filter(PnT, .data$Type=='Peak') %>%
        select(-.data$Type)
    }
    return(PnT)
  }

  # Otherwise, calculate differences and slope of vectors  groups of
  # between peaks and troughs


  PnTAbr <- PnT %>%
    mutate(dX = .data$X - lag(.data$X), dY = .data$Y - lag(.data$Y),
           # Get differences in x and y between adjacent
           # x, y pairs
           grp = (.data$dY ^ 2) < (minSlope*.data$dX)^2 + minYerror^2,
           # Test if pair is too close to be considered separate
           # using filtering conditions
           grp=ifelse(or(.data$grp, is.na(.data$grp)), 0, 1),
           group=sum_line(.data$grp)) %>%
    group_by(.data$group) %>%
    mutate(grp = .data$grp +
             write_groups(.data$X, .data$Y, minSlope, minYerror,
                          globalFilter = globalFilter)) %>%
    ungroup() %>%
    mutate(group=sum_line(.data$grp)) %>%
    group_by(.data$group) %>%
    # Each group consists of a set of peaks and troughs
    # which do not stand out from each other.
    # Rename the type of each point to be the type of its group
    mutate(Type = ifelse(first(.data$Type) != last(.data$Type), 'None',
                         first(.data$Type)),
           Type = as.factor(.data$Type)) %>%
    # Filter groups which are neither peaks nor troughs
    # Filter all but the 'outstanding points': the highest
    # points in the peak groups and the lowest in the trough groups
    filter(.data$Type != 'None', last(.data$X) - first(.data$X) < maxPeakWidth,
           .data$Y == ifelse(.data$Type == 'Peak',
                                       max(.data$Y),
                                       min(.data$Y))) %>%
    slice_head() %>% # Keep only the first outstanding point in each group
    ungroup() %>%
    select(-.data$dX, -.data$dY, -.data$group, -.data$grp)
  if (justPeaks){
    PnTAbr <- filter(PnTAbr, .data$Type == 'Peak') %>%
      select(-.data$Type)
  }
  if (edgeFilter > 0){
    PnTAbr <- filter(PnTAbr, .data$X > xMin + edgeFilter, .data$X < xMax - edgeFilter)
  }
return(PnTAbr)


}

utils::globalVariables(c('.'))

#' Create array of running totals
#'
#' \code{sum_line} creates an array of running totals
#' from an integer array
#'
#' This function takes an array of integers and
#' returns an array with each entry containing
#' the sum of that entry and all previous entries
#' in the input array.
#'
#' @param arr The array of integers
#'
#' @return An integer array of running totals.
sum_line <- function(arr){
  A <- vector('integer', length(arr))
  val = 0
  for (x in 1:length(arr)){
    val <- val + arr[x]
    A[x] <- val
  }
  return(A)
}

# write_groups <- function(X, Y, minSlope, minYerror, globalFilter = TRUE) {
#
#   # If not using the globalFilter, return a vector of zeros
#   if (!globalFilter) {return(rep(0, length(X)))}
#
#   newgroup <- TRUE
#   res <- vector('integer', length(X))
#   for (i in 1:length(X)){
#     if (newgroup){ # If the current point is the start of a new group, move
#       # all rectangle boundaries to the position of that point.
#       minX <- X[i]
#       maxX <- minX
#       minY <- Y[i]
#       maxY <- minY
#       res[i] <- 0
#       newgroup <- FALSE
#
#     } else { # Update rectangle boundaries based on position of current point
#
#       maxX <- X[i]
#       if (Y[i] > maxY){
#         maxY <- Y[i]
#       } else if (Y[i] < minY){
#         minY <- Y[i]
#       }
#       dY <- maxY - minY
#       dX <- maxX - minX # Find dimensions of rectangle
#       # Test if rectangle fits the grouping parameters
#       group <- (dY ^ 2) < (minSlope*dX)^2 + minYerror^2
#       # If the rectangle is too large, start a new group
#       res[i] <- ifelse(group, 0, 1)
#       if (!group){
#         newgroup <- TRUE
#       }
#     }
#   }
#
#   return(res)
# }

