% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmm.R
\name{lmm}
\alias{lmm}
\title{Fitting univariate and multiviarate linear mixed models via data transforming augmentation}
\usage{
lmm(y, v, x = 1, n.burn, n.sample, tol = 1e-10,
  method = "em", dta = TRUE, print.time = FALSE)
}
\arguments{
\item{y}{Response variable. In a univariate case, it is a vector of length \eqn{k} for the observed data. In a multivariate case, it is a (\eqn{k} by \eqn{p}) matrix, where \eqn{k} is the number of observations and \eqn{p} denotes the dimensionality.}

\item{v}{Known measurement error variance. In a univariate case, it is a vector of length \eqn{k}. In a multivariate case, it is a (\eqn{p}, \eqn{p}, \eqn{k}) array of known measurement error covariance matrices, i.e., each of the \eqn{k} array components is a (\eqn{p} by \eqn{p}) covariance matrix.}

\item{x}{(Optional) Covariate information. If there is one covariate for each object, e.g., weight, it is a  vector of length \eqn{k} for the weight. If there are two covariates for each object, e.g., weight and height, it is a (\eqn{k} by 2) matrix, where each column contains a covariate variable. Default is no covariate (\code{x = 1}).}

\item{n.burn}{Number of warming-up iterations for a Markov chain Monte Carlo method. It must be specified for \code{method = "mcmc"}}

\item{n.sample}{Number of iterations (size of a posterior sample for each parameter) for a Markov chain Monte Carlo method. It must be specified for \code{method = "mcmc"}}

\item{tol}{Tolerance that determines the stopping rule of the EM algorithm. The EM algorithm iterates until the change of log-likelihood function is within the tolerance. Default is 1e-10.}

\item{method}{\code{"em"} will return maximum likelihood estimates of the unknown hyper-parameters and \code{"mcmc"} returns posterior samples of those parameters.}

\item{dta}{A logical; Data transforming augmentation is used if \code{dta = TRUE}, and typical data augmentation is used if \code{dta = FALSE}.}

\item{print.time}{A logical; \code{TRUE} to display two time stamps for initiation and termination, \code{FALSE} otherwise.}
}
\value{
The outcome of \code{lmm} is composed of:
\describe{
\item{A}{If \code{method} is "mcmc". It contains the posterior sample of \eqn{A}.}
\item{beta}{If \code{method} is "mcmc". It contains the posterior sample of \eqn{\beta}.}
\item{A.mle}{If \code{method} is "em". It contains the maximum likelihood estimate of \eqn{A}.}
\item{beta.mle}{If \code{method} is "em". It contains the maximum likelihood estimate of \eqn{beta}.}
\item{A.trace}{If \code{method} is "em". It contains the update history of \eqn{A} at each iteration.}
\item{beta.trace}{If \code{method} is "em". It contains the update history of \eqn{beta} at each iteration.}
\item{n.iter}{If \code{method} is "em". It contains the number of EM iterations.}
}
}
\description{
The function \code{lmm} fits univariate and multivariate linear mixed models
 (also called two-level Gaussian hierarchical models) whose first-level hierarchy is about
 a distribution of observed data and second-level hierarchy is about a prior distribution of random effects.
}
\details{
 For each group \eqn{i}, let \eqn{y_{i}} be an unbiased estimate of random effect \eqn{\theta_{i}},
 and \eqn{V_{i}} be a known measurement error variance. The linear mixed model of interest is specified as follows:
 \deqn{[y_{i} \mid \theta_{i}] \sim N(\theta_{i}, V_{i})}
 \deqn{[\theta_{i} \mid \mu_{0i}, A) \sim N(\mu_{0i}, A)}
 \deqn{\mu_{0i} = x_{i}'\beta} 
independently for \eqn{i = 1, \ldots, k}, where \emph{k} is the number of groups (units) and dimension of each element is appropriately adjusted in a multivariate case.

The function \code{lmm} produces maximum likelihood estimates of hyper-parameters, \eqn{A} and \eqn{\beta}, their update histories of EM iterations, and the number of EM iterations if \code{method} is \code{"em"}. 

For a Bayesian implementation, we put a jointly uniform prior distribution on \eqn{A} and \eqn{\beta}, i.e.,
 \deqn{f(A, \beta) \propto 1,}
which is known to have good frequency properties. This joint prior distribution is improper, but their resulting posterior distribution is proper if \eqn{k\ge m+p+2}, where \eqn{k} is the number of groups, \eqn{m} is the number of regression coefficients, and \eqn{p} is the dimension of \eqn{y_{i}}. We note that an R package \code{Rgbp} also fits this model in a univariate case (\eqn{p=1}) via ADM (approximation for density maximization). \code{lmm} produces the posterior samples through a Gibbs sampler if \code{method} is \code{"bayes"}.
}
\examples{
### Univariate linear mixed model

# response variable for 10 objects
y <- c(5.42, -1.91, 2.82, -0.14, -1.83, 3.44, 6.18, -1.20, 2.68, 1.12)
# corresponding measurement error standard deviations
se <- c(1.05, 1.15, 1.22, 1.45, 1.30, 1.29, 1.31, 1.10, 1.23, 1.11)
# one covariate information for 10 objects
x <- c(2, 3, 0, 2, 3, 0, 1, 1, 0, 0)

## Fitting without covariate information
# (DTA) maximum likelihood estimates of A and beta via an EM algorithm
res <- lmm(y = y, v = se^2, method = "em", dta = TRUE)
# (DTA) posterior samples of A and beta via an MCMC method
res <- lmm(y = y, v = se^2, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = TRUE)
# (DA) maximum likelihood estimates of A and beta via an EM algorithm
res <- lmm(y = y, v = se^2, method = "em", dta = FALSE)
# (DA) posterior samples of A and beta via an MCMC method
res <- lmm(y = y, v = se^2, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = FALSE)

## Fitting with the covariate information
# (DTA) maximum likelihood estimates of A and beta via an EM algorithm
res <- lmm(y = y, v = se^2, x = x, method = "em", dta = TRUE)
# (DTA) posterior samples of A and beta via an MCMC method
res <- lmm(y = y, v = se^2, x = x, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = TRUE)
# (DA) maximum likelihood estimates of A and beta via an EM algorithm
res <- lmm(y = y, v = se^2, x = x, method = "em", dta = FALSE)
# (DA) posterior samples of A and beta via an MCMC method
res <- lmm(y = y, v = se^2, x = x, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = FALSE)


### Multivariate linear mixed model

# (arbitrary) 10 hospital profiling data (two response variables)
y1 <- c(10.19, 11.53, 16.28, 12.32, 12.84, 11.85, 14.81, 13.24, 14.43, 9.35)
y2 <- c(12.06, 14.97, 11.50, 17.88, 19.21, 14.69, 13.96, 11.07, 12.71, 9.63)
y <- cbind(y1, y2)

# making measurement error covariance matrices for 10 hospitals
n <- c(24, 34, 38, 42, 49, 50, 79, 84, 96, 102) # number of patients
v0 <- matrix(c(186.87, 120.43, 120.43, 250.60), nrow = 2) # common cov matrix
temp <- sapply(1 : length(n), function(j) { v0 / n[j] })
v <- array(temp, dim = c(2, 2, length(n)))

# covariate information (severity measure)
severity <- c(0.45, 0.67, 0.46, 0.56, 0.86, 0.24, 0.34, 0.58, 0.35, 0.17)

## Fitting without covariate information
# (DTA) maximum likelihood estimates of A and beta via an EM algorithm
\donttest{
res <- lmm(y = y, v = v, method = "em", dta = TRUE)
}
# (DTA) posterior samples of A and beta via an MCMC method
\donttest{
res <- lmm(y = y, v = v, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = TRUE)
}
# (DA) maximum likelihood estimates of A and beta via an EM algorithm
\donttest{
res <- lmm(y = y, v = v, method = "em", dta = FALSE)
}
# (DA) posterior samples of A and beta via an MCMC method
\donttest{
res <- lmm(y = y, v = v, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = FALSE)
}

## Fitting with the covariate information
# (DTA) maximum likelihood estimates of A and beta via an EM algorithm
\donttest{
res <- lmm(y = y, v = v, x = severity, method = "em", dta = TRUE)
}
# (DTA) posterior samples of A and beta via an MCMC method
\donttest{
res <- lmm(y = y, v = v, x = severity, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = TRUE)
}
# (DA) maximum likelihood estimates of A and beta via an EM algorithm
\donttest{
res <- lmm(y = y, v = v, x = severity, method = "em", dta = FALSE)
}
# (DA) posterior samples of A and beta via an MCMC method
\donttest{
res <- lmm(y = y, v = v, x = severity, n.burn = 1e1, n.sample = 1e1,
           method = "mcmc", dta = FALSE)
}
}
\references{
Tak, You, Ghosh, Su, Kelly (2019), "Data Transforming Augmentation for Heteroscedastic Models" <\doi{10.1080/10618600.2019.1704295}>
}
\author{
Hyungsuk Tak (maintainer), Kisung You, Sujit K. Ghosh, and Bingyue Su
}
