#' Create masking layers for the depth layers
#'
#' \code{createDepthLayerMasks} - Creates a masking layer, i.e., a
#' true/false-matrix, indicating which pixels of an image belong to a specified
#' depth layer.\cr
#' We assume the image was taken inside a minirhizotron, a round
#' plexiglas tube used to observe roots in the ground, and shows a 360 degree
#' scan of a section of the tube. As the minirhizotron is installed at an
#' angle (can be specified, by default 45 degrees) a depth layer in the ground
#' is shown as a sinus curve in the scan.
#'
#' @param dims_px Integer vector of length two
#' specifying the dimensions of the image/mask layer (in pixels).
#' @param ppcm Numeric value specifying how many pixels there are per
#' centimeter (resolution of the image). Default NULL.
#' If \code{ppcm} is not NULL or NA, \code{ppi} is ignored.\cr
#' Examples: 300 ppi (printing) is 118 ppcm, 150 ppi is 59 ppcm,
#' 72 ppi (screens) is 28 ppcm.
#' @param ppi Numeric value specifying how many pixels there are per inch
#' (resolution of the image). Default NULL.
#' Leave/set \code{ppcm} to NULL or NA to use \code{ppi}.\cr
#' Examples: 300 ppi (printing) is 118 ppcm, 150 ppi is 59 ppcm,
#' 72 ppi (screens) is 28 ppcm.
#' @param depth_levels_cm  Numeric matrix with two columns and at least
#' one row. Each row specifies a depth interval (in cm) that is of
#' interest. For each interval a masking layer is generated. The first value
#' of each row has to be larger than the second (decending).\cr
#' By default: Depth layers Depth layers 0-(-10)cm, -10-(-20)cm, -20-(-30)cm,
#' and -30-(-40)cm.
#' @param depth_highpoint_cm Numeric value specifying the depth at which the
#' highest point of the image is (in cm).
#' By default this is at depth 0.\cr
#' More precisely, this refers to the depth of the middle of the pixel in the
#' first row of the \code{top_side} of the image.
#' @param pos_highpoint_px Either one of "center" or "edge" (default),
#' indicating that scanning starts at the bottom or top facing side of the
#' minirhizotron, respectively, or it can also be an integer value (>=1 and
#' <=width of the \code{top_side} of the image) specifying where (left<->right)
#' in the top row of the image the highest point of the image lies, indicating
#' the column where the minimum points of the depth-level lines lie. \cr
#' "edge" is equivalent to using 1 or width of the \code{top_side} and
#' "center" to using half the width of the \code{top_side}.\cr
#' See also \code{gap_cm} if there is a non-zero gap.
#' @param angle Numeric value >=0 and <=90 (default 45) specifying the
#' installation angle of the minirhizotron in degrees (angle between the ground
#' and the tube above the soil).
#' @param top_side One of "left","top" (default),"right","bottom". Indicates
#' where the upper side of the image is.
#' @param gap_cm Numeric value (default 0) that specifies if there is a gap
#' (in cm) between the two "matching" sides of the image, i.e., there is no full
#' rotation of the scanner. If \code{pos_highpoint_px} is specified as "center"
#' or "edge" (in characters), then it is assumed that the middle of the gap is
#' right at the top or bottom facing side of the tube, i.e. it is not possible
#' to start and end at the exact top/bottom. If \code{pos_highpoint_px} is
#' specified as a numeric value, then it is considered as exact and the gap
#' does not move it.
#' @param gap_deg Numeric value (default 0) that specifies if there is a gap
#' (in degrees, i.e. from 0 to 360) between the two "matching" sides
#' of the image, i.e., there is no full rotation of the scanner.
#' Works similar to \code{gap_cm}. Example: If the scanner scans only 355
#' degrees, then set the value to 5. \code{gap_deg} is only used if
#' \code{gap_cm} is NA.
#'
#' @return \code{createDepthLayerMasks} A list of 2-dimensional
#' true/false-matrices with the specified dimensions (see \code{dims}). True
#' indicates that the corresponding pixel belongs to the specified depth layer.
#' The list contains \code{nrow(depth_levels_cm)}-many matrices, one for each
#' specified depth layer.
#'
#' @export
#' @rdname createDepthLayerMasks
#'
#' @examples
#' # Small example for creating a masking layer of a single depth interval:
#' createDepthLayerMasks(ppcm = 1, dims_px = c(10,10),
#'                       depth_levels_cm = matrix(c(-3,-5), ncol = 2))[[1]]
#' # Change some parameters to adapt to a different minimum point position and
#' # installation angle.
#' createDepthLayerMasks(ppcm = 1, dims_px = c(5,7),
#'                       depth_levels_cm = matrix(c(-1.5,-2.5), ncol = 2),
#'                       pos_highpoint_px = "center", gap_cm = 0)[[1]]
#' # And here is a cutout of that mask using a gap.
#' createDepthLayerMasks(ppcm = 1, dims_px = c(5,5),
#'                       depth_levels_cm = matrix(c(-1.5,-2.5), ncol = 2),
#'                       pos_highpoint_px = "center", gap_cm = 2)[[1]]
createDepthLayerMasks <- function(ppcm = NULL, ppi = NULL, dims_px,
                                  depth_levels_cm = rbind(c(0,-10),
                                                          c(-10,-20),
                                                          c(-20,-30),
                                                          c(-30,-40)),
                                  depth_highpoint_cm = 0,
                                  pos_highpoint_px = "center", angle = 45,
                                  top_side = "top",
                                  gap_cm = 0, gap_deg = 0){
  # Check input. ---------------------------------------------------------------
  if(length(dims_px)!=2 || length(depth_levels_cm)<2){
    stop("Check the length/size of 'dims_px' and 'depth_levels_cm'.")
  }
  if(is.null(ppcm) || is.na(ppcm)){
    if(!is.null(ppi)){
      ppcm <- ppi2ppcm(ppi)
    } else {
      stop("No resolution specified ('ppcm' and 'ppi' are both NULL).")
    }
  }
  if(sum(c(ppcm, dims_px, gap_cm)<0, na.rm = TRUE)>0){
    stop(paste("No negative values allowed for 'ppcm', 'ppi', 'dims_px', and",
               "'gap_cm'."))
  }
  if(sum(depth_levels_cm[,1] <= depth_levels_cm[,2])>0){
    stop("Each row of 'depth_levels_cm' have to be sorted in decreasing order.")
  }
  if(angle<0 || angle>90){
    stop("The angle must be >=0 and <=90.")
  }
  if(gap_deg<0 || gap_deg>=360){
    stop("'gap_deg' must be >=0 and <360.")
  }
  if(!top_side %in% c("top", "left", "right", "bottom")){
    stop("Unknown 'top_side'.")
  }
  if(!is.numeric(depth_highpoint_cm)){
    stop("'depth_highpoint_cm' not numeric.")
  }
  # Prepare some parameters. ---------------------------------------------------
  # Depending on where the top_side is, change the dimensions of the images.
  if(top_side == "left" || top_side == "right"){
    dims_px <- rev(dims_px)
  }

  # If there is a gap only given in degrees compute the corresponding gap in
  # pixels.
  if(is.na(gap_cm)){
    if(is.na(gap_deg)){
      gap_px <- 0
    } else {
      img_width <- dims_px[2] # corresponds to 360-gap_deg
      img_width_360 <- img_width * 360 / (360-gap_deg)
      gap_px <- round(img_width_360 - img_width)
    }
  } else {
    gap_px <- round(cm2px(gap_cm, ppcm = ppcm))
  }
  # If there is a gap, temporarily increase the dimensions to later cut the
  # corresponding part out again. On the left: floor(gap_px/2) columns added,
  # on the right: ceiling(gap_px/2) many columns added.
  dims_px <- dims_px + c(0, gap_px)

  # Where is the highpoint/upwards facing side of the tube in the images?
  if(pos_highpoint_px == "center"){
    pos_highpoint_px <- round(dims_px[2]/2) # Still in the center even with gap.
  } else if(pos_highpoint_px == "edge"){
    pos_highpoint_px <- 1 # Still at the edge even with gap.
  } else {
    # Here shift it to the right by what was added at the left.
    pos_highpoint_px <- pos_highpoint_px + floor(gap_px/2)
    if(pos_highpoint_px<1){
      stop(paste0("Only positive values allowed for",
                  "'pos_highpoint_px'+floor(gap_px/2)."))
    }
  }

  # Create the masking layers. -------------------------------------------------
  list_of_masks <- list()
  for(lvl_index in 1:nrow(depth_levels_cm)){
    # Initialize a masking layer.
    mask <- matrix(FALSE, nrow = dims_px[1], ncol = dims_px[2])
    # Sketch for depth interval (a,b) with minimum points u and l (pixel
    # coordinates) and the minimum point A of the whole image:
    # Rows\Columns:
    #      1 2 3 4 ....            m
    #    |__________________________|                                 O\
    # 1  |         (A1,A2)          |         Sketch minirhizotron :  \ \
    # 2  |----                   ---|                                  \ \
    # 3  |    \                /    |   <- dissection line for depth a -\-\-
    # 4  |     `---(u1,u2)---´      |                                    \ \
    # .  |----                   ---|                                     \ \
    # .  |    \                /    |   <- dissection line for depth b    -\-\-
    # .  |     `---(l1,l2)---´      |                                       \ \
    # n  |                          |                                        \ \
    #    |__________________________|                                         \O
    #
    #   ^(0,n-1)
    #   |  This is the "new" coordinate system in which we compute
    #   |  everything.
    #   |
    #   |(0,0)  ...   (m-1,0)
    #   ___________________>
    # Compute coordinates of the minimum point u of the upper dissection line.
    # Convert degrees into radians.
    depth2height <- function(depth_diff){
      return(depth_diff/sin(angle/180 * pi))
    }
    px_coord_up <- c(1 + round(cm2px(depth2height(depth_highpoint_cm-
                                                 depth_levels_cm[lvl_index,1]),
                                    ppcm = ppcm)),
                    pos_highpoint_px)

    px_dist_to_low <- round(cm2px(depth2height(depth_levels_cm[lvl_index,1]-
                                                 depth_levels_cm[lvl_index,2]),
                                  ppcm = ppcm))
    # Convert them to the new coordinate system.
    old2newCoords <- function(old_coords){
      return(c(old_coords[2] - 1, dims_px[1] - old_coords[1]))
    }
    px_coord_new_up <- old2newCoords(px_coord_up)

    # For every x from 0 to m-1 compute the dissection line values y.
    x_new <- 0:(dims_px[2]-1)
    y_upper_new <- round(sapply(X=x_new, function(X){
      dissectionLine(x=X, minpoint = px_coord_new_up, angle = angle,
                     radius = (dims_px[2]-1)/(2*pi))}))

    # Convert these values back to the old coordinate system.
    col_vals <- x_new + 1
    row_vals_upper <- dims_px[1] - y_upper_new

    # For each column set the values to TRUE below the upper values to create
    # a TRUE-ribbon of width=px_dist_to_low.
    for(col_index in 1:dims_px[2]){
      rows_to_true <- row_vals_upper[col_index]+ (0:px_dist_to_low)
      rows_to_true <- rows_to_true[rows_to_true>=1 & rows_to_true <= dims_px[1]]
      mask[rows_to_true, col_index] <- TRUE
    }

    # If there is a gap, cut the the temporary part out again.
    if(gap_px>0){
      col_to_remove <- c(1:floor(gap_px/2),
                         (ncol(mask)-ceiling(gap_px/2)+1):ncol(mask))
      mask <- mask[,-col_to_remove]
    }

    list_of_masks[[lvl_index]] <- mask
  }
  # Depending on where the top_side is. Rotate the matrices.
  if(top_side == "left"){ # Rotate the matrix
    list_of_masks <- lapply(1:length(list_of_masks),
                          function(X){
                            return(apply(t(list_of_masks[[X]]), 2, rev))
                          })
  } else if(top_side == "right"){
    list_of_masks <- lapply(1:length(list_of_masks),
                          function(X){
                            return(t(apply(list_of_masks[[X]], 2, rev)))
                          })
  } else if(top_side == "bottom"){
    list_of_masks <- lapply(1:length(list_of_masks),
                          function(X){
                            return(t(apply(t(apply(list_of_masks[[X]], 2, rev)),
                                           2, rev)))
                          })
  }
  return(list_of_masks)
}
#' Create masking layers for the depth layers
#'
#' \code{dissectionLine} - Returns the y-value of the dissection line of a
#' tube that was positioned at an angle for a given position x.
#' The dissection line is dependent on the \code{minpoint}, the \code{angle},
#' and the \code{radius} of the tube (have to be specified).
#'
#' @param x Numeric value indicating the position.
#' @param minpoint Numeric vector of length 2 indicating the coordinates of the
#' minimum point of the function.
#' @param radius Positive numeric value indicating the radius of the tube.
#'
#' @return \code{dissectionLine}
#'
#' @export
#' @rdname createDepthLayerMasks
#'
#' @examples
#' # Examples how different angles result in different dissection graphs:
#' test_x <- -12:12
#' plot(test_x, sapply(X=test_x, function(X){
#'                dissectionLine(x=X, minpoint = c(0,0), angle = 45,
#'                               radius = 3)}),
#'      ylim = c(0, 20))
#' plot(test_x, sapply(X=test_x, function(X){
#'                dissectionLine(x=X, minpoint = c(0,0), angle = 80,
#'                               radius = 3)}),
#'      ylim = c(0, 20))
#' plot(test_x, sapply(X=test_x, function(X){
#'                dissectionLine(x=X, minpoint = c(0,0), angle = 10,
#'                               radius = 3)}),
#'      ylim = c(0, 20))
dissectionLine <- function(x, minpoint, angle, radius){
  if(angle<0 || angle>90){
    stop("The angle must be >=0 and <=90.")
  }
  # Convert angle in degree to radians:
  angle <- angle/180 * pi
  # Compute y-value:
  y <- radius * tan(pi/2 - angle) * (1 - cos((x-minpoint[1])/radius)) +
    minpoint[2]
  return(y)
}
