% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sa.tabu.R
\name{sa.tabu}
\alias{sa.tabu}
\title{Sensitivity Analysis for SEM using Tabu Search}
\usage{
sa.tabu(
  model,
  sens.model,
  data = NULL,
  sample.cov = NULL,
  sample.nobs = NULL,
  opt.fun = 1,
  sig.level = 0.05,
  ...
)
}
\arguments{
\item{model}{The analytic model of interest.}

\item{sens.model}{Sensitivity analysis model template for
structural equation modeling
with a phantom variable. This is the model of interest
with a phantom variable and sensitivity parameters added.
See examples provided.}

\item{data}{The data set used for analysis.}

\item{sample.cov}{covariance matrix for SEM analysis
when data are not available.}

\item{sample.nobs}{Number of observations for covariance matrix.}

\item{opt.fun}{Customized or preset optimization function.
The argument can be customized as a function, e.g., opt.fun =
quote(new.par$pvalue[paths]-old.par$pvalue[paths]), where new.par and old.par
are the parameter estimates from the sensitivity analysis and analytic models,
respectively.
When opt.fun is
1, the optimization function is the average departure of new estimate
from the old estimate divided by the old estimate
y <-  mean(abs(new.par$est[paths] -
old.par$est[paths]))/mean(abs(old.par$est[paths])); When opt.fun is
2, the optimization function is the standard deviation of deviance
divided by the old estimate
y <-  stats::sd(new.par$est[paths] - old.par$est[paths])/
mean(abs(old.par$est[paths]));
When opt.fun is 3, the optimization function is the average
p value changed or
y <-  mean(abs(new.par$pvalue[paths] - old.par$pvalue[paths]));
When opt.fun is 4, the optimization function is the average distance
from significance level or y <-  mean(abs(new.par$pvalue[paths] -
rep(sig.level,length(paths))));
When opt.fun is 5, we assess the change of RMSEA or
y <-  abs(unname(lavaan::fitmeasures(new.out)["rmsea"]) -
unname(lavaan::fitmeasures(old.out)["rmsea"]));
When opt.fun is 6, we optimize how close RMSEA is to 0.05 or
y <-  1/abs(unname(lavaan::fitmeasures(new.out)["rmsea"]) - 0.05).}

\item{sig.level}{Significance level, default value is 0.05.}

\item{...}{Additional arguments from the lavaan package.}
}
\value{
A list with five components:
    model: The old model;
    old.model.par: Parameters of the old model;
    model.results: Sensitivity analysis model results;
    best.param: Parameters that optimize the objective function;
    best.obj: The optimized objective function value;
    sens.par: NULL. Included for compatibility;
    outcome: NULL. Included for compatibility.
}
\description{
This function conducts sensitivity analysis
    for SEM using tabu search.
}
\examples{
library(lavaan)
# Generate data, this is optional as lavaan also takes variance covariance matrix
sim.model <- ' x =~ x1 + 0.8*x2 + 1.2*x3
               y =~ y1 + 0.5*y2 + 1.5*y3
               m ~ 0.5*x
               y ~ 0.5*x + 0.8*m'
set.seed(10)
data <- simulateData(sim.model, sample.nobs = 1000L)
# standardize dataset
data = data.frame(apply(data,2,scale))

# Step 1: Set up the analytic model of interest
model <- 'x =~ x1 + x2 + x3
          y =~ y1 + y2 + y3
          m ~ x
          y ~ x + m'

# Step 2: Set up the sensitivity analysis model.
#         The sensitivity parameters are phantom1, phantom2, and phantom3 in this example.
sens.model = 'x =~ x1 + x2 + x3
              y =~ y1 + y2 + y3
              m ~ x
              y ~ x + m
              x ~ phantom1*phantom
              m ~ phantom2*phantom
              y ~ phantom3*phantom
              phantom =~ 0 # added for mean of zero
              phantom ~~ 1*phantom' # added for unit variance
# Step 3: Set up the paths of interest to be evaluated in sensitivity analysis.
# Suppose we are interested in all direct and indirect paths.
  paths <- 'm ~ x
            y ~ x + m'
# Step 4: Perform sensitivity analysis
out <- sa.tabu(model = model,
               sens.model = sens.model,
               data = data,
               opt.fun = 1,
               max.iter = 2,
               max.iter.obj = 2)
 # Note, please specify larger numbers for
 # max.iter (e.g., 50) and max.iter.obj (e.g., 10)

# Step 5: Summarize sensitivity analysis results.
# See sens.tables function for explanation of results.
tables <- sens.tables(out)

}
