% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chi2_test.R
\name{chi2.test}
\alias{chi2.test}
\title{Chi-squared test for first-order separability of a spatio-temporal point process}
\usage{
chi2.test(
  X,
  sim.procedure = c("pure_per", "block_per"),
  nblocks = 5L,
  nperm = 199L,
  n.time = 2L,
  n.space = 3L,
  t.region = c(0, 1),
  s.region = c(0, 1, 0, 1)
)
}
\arguments{
\item{X}{A numeric matrix or data frame with at least three columns giving event coordinates
\eqn{(x, y, t)}.}

\item{sim.procedure}{Character string specifying the procedure: \code{"pure_per"} or \code{"block_per"}.}

\item{nblocks}{Integer (>= 2). Number of temporal blocks used for block permutation (only for \code{"block_per"}).}

\item{nperm}{Integer (>= 1). Number of Monte Carlo permutations (only for \code{"block_per"}).}

\item{n.time}{Integer (>= 2). Number of temporal intervals in the contingency table.}

\item{n.space}{Integer (>= 2). The spatial domain is partitioned into \code{n.space} bins per axis
(yielding \code{n.space^2} spatial cells) for the contingency table.}

\item{t.region}{Numeric vector of length 2 giving the temporal window \code{c(tmin, tmax)} with \code{tmin < tmax}.}

\item{s.region}{Spatial window specification. By default, the bounding box \code{c(0, 1, 0, 1)}
corresponding to \code{c(xmin, xmax, ymin, ymax)}. Passed to \code{\link{chisq.test.stPP}}.}
}
\value{
Numeric scalar: the p-value of the test.
}
\description{
Performs a chi-squared test for testing first-order separability of a spatio-temporal point process.
Two procedures are available:
\describe{
\item{\code{"pure_per"}}{Classical asymptotic chi-squared test of independence on a space--time count table.}
\item{\code{"block_per"}}{Monte Carlo permutation test based on block-wise permutations of the time component.}
}
}
\details{
The classical procedure (\code{"pure_per"}) applies a chi-squared test of independence to the
\code{n.space^2} by \code{n.time} contingency table of counts.

The permutation procedure (\code{"block_per"}) generates  up to \code{nperm} block-permuted datasets under the null
using \code{\link{sim.procedures}} with \code{method = "block"}, recomputes the chi-squared statistic for each,
and returns a Monte Carlo p-value computed as \eqn{(1 + \#\{T_i \ge T_{obs}\})/(nperm + 1)}.
}
\examples{

set.seed(124)
lambda <- get.lambda.function(N = 200, g = 50, model = 4)
Lmax <- get.lambda.max(N = 200, g = 50, model = 4)
X <- rstpoispp(lambda, Lmax)


# Classical chi-squared test
chi2.test(X, sim.procedure = "pure_per", n.time = 2, n.space = 3)

# Monte Carlo permutation test with blocks
chi2.test(X, sim.procedure = "block_per", nblocks = 5, nperm = 100)

}
\references{
Ghorbani M., Vafaei N., Dvořák J., Myllymäki M. (2021).
Testing the first-order separability hypothesis for spatio-temporal point patterns.
\emph{Computational Statistics and Data Analysis}, \bold{161}, 107245.

Ghorbani, M., Vafaei, N. and Myllymäki, M. (2025).
A kernel-based test for the first-order separability of spatio-temporal point processes,
\emph{TEST}.
}
\seealso{
\code{\link{chisq.test.stPP}}, \code{\link{sim.procedures}}, \code{\link{block.permut}}
}
\author{
Mohammad Ghorbani \email{mohammad.ghorbani@slu.se}\cr
Nafiseh Vafaei \email{nafiseh.vafaei@slu.se}
}
