% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OptStiefelGBB.R
\name{OptStiefelGBB}
\alias{OptStiefelGBB}
\title{Optimization on Stiefel manifold}
\usage{
OptStiefelGBB(X, fun, opts = NULL, ...)
}
\arguments{
\item{X}{Initial value to start the optimization. A \eqn{n} by \eqn{k} orthonormal matrix such that \eqn{X^T X = I_k}.}

\item{fun}{The function that returns the objective function value and its gradient. The syntax for \code{fun} is \code{fun(X, data1, data2)} where \code{data1, data2} are additional data passed to \code{...}.}

\item{opts}{A list specifying additional user-defined arguments for the curvilinear search algorithm. Some important ones are listed in the following:
\itemize{
 \item \code{maxiter}: The maximal number of iterations.
 \item \code{xtol}: The convergence tolerance for \eqn{X}, e.g., \eqn{||X^{(t)} - X^{(t-1)}||_F/\sqrt{k}}.
 \item \code{gtol}: The convergence tolerance for the gradient of the Lagrangian function, e.g., \eqn{||G^{(t)} - X^{(t)} (G^{(t)})^T X^{(t)}||_F}.
 \item \code{ftol}: The convergence tolerance for objective function \eqn{F}, e.g., \eqn{|F^{(t)} - F^{(t-1)}|/(1+|F^{(t-1)}|)}. Usually, \code{max{xtol, gtol} > ftol}.
}
The default values are: \code{maxiter=500; xtol=1e-08; gtol=1e-08; ftol=1e-12.}}

\item{...}{Additional input passed to \code{fun}.}
}
\value{
\item{X}{The converged solution of the optimization problem.}
\item{out}{Output information, including estimation error, function value, iteration times etc.
\itemize{
\item{\code{nfe}}: The total number of line search attempts.
\item{\code{msg}}: Message: "convergence" | "exceed max iteration".
\item{\code{feasi}}: The feasibility of solution: \eqn{||X^TX - I_k||_F}.
\item{\code{nrmG}}: The convergence criterion based on the projected gradient \eqn{||G - X G^T X||_F}.
\item{\code{fval}}: The objective function value \eqn{F(X)} at termination.
\item{\code{iter}}: The number of iterations.
}}
}
\description{
Curvilinear search algorithm for optimization on Stiefel manifold developed by Wen and Yin (2013).
}
\details{
The calling syntax is \code{OptStiefelGBB(X, fun, opts, data1, data2)}, where \code{fun(X, data1, data2)} returns the objective function value and its gradient.

For example, for \eqn{n} by \eqn{k} matrix \eqn{X}, the optimization problem is
\deqn{min_{X} -tr(X^T W X), \mbox{ such that } X^T X = I_k.}
The objective function and its gradient are
\deqn{F(X) = -tr(X^T W X), \; G(X) = - 2 W X.}
Then we need to provide the function \code{fun(X, W)} which returns \eqn{F(X)} and \eqn{G(X)}. See \strong{Examples} for details.

For more details of the termination rules and the tolerances, we refer the interested readers to Section 5.1 of Wen and Yin (2013).
}
\examples{
n <- 1000
k <- 6

# Randomly generated matrix M
W <- matrix(rnorm(n^2), n, n)
W <- t(W) \%*\% W

# Randomly generated orthonormal initial matrix
X0 <- matrix(rnorm(n*k), n, k)
X0 <- qr.Q(qr(X0))

# The objective function and its gradient
fun <- function(X, W){
  F <- - sum(diag(t(X) \%*\% W \%*\% X))
  G <- - 2*(W \%*\% X)
  return(list(F = F, G = G))
}

# Options list
opts<-list(record = 0, maxiter = 1000, xtol = 1e-5, gtol = 1e-5, ftol = 1e-8)

# Main part
output <- OptStiefelGBB(X0, fun, opts, W)
X <- output$X
out <- output$out

}
\references{
Wen, Z. and Yin, W., 2013. A feasible method for optimization with orthogonality constraints. Mathematical Programming, 142(1-2), pp.397-434.
}
