#' @title Symmetry check
#' @description Given a game and two players, this function checks if those are symmetric players.
#' @param v A characteristic function, as a vector.
#' @param i The position of an individual coalition, as an integer.
#' @param j The position of another individual coalition, as an integer.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v}, \code{i} and \code{j} are introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number.\cr
#'            By default, \code{tol=100*.Machine$double.eps}.
#' @return \code{TRUE} if \code{i} and \code{j} are symmetric in \code{v}, \code{FALSE} otherwise.
#' @details Let \eqn{v\in G^N}. Players \eqn{i,j \in N} are symmetric in \eqn{v} if,
#' for each \eqn{S \subset N} with \eqn{i,j \in S}, \eqn{v(S \setminus \{i\}) = v(S \setminus \{j\})}.
#' @examples
#' symmetrycheck(c(13,13,0,0,58,58,0),1,2) # players 1 and 2 are symmetric
#' @export

symmetrycheck <- function(v, i, j, binary = FALSE, tol = 100*.Machine$double.eps) {

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.

  if (log2(length(v) + 1) %% 1 != 0) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (binary == FALSE) {
    if (i %in% 1:n == FALSE) {
      stop("'i' must be the position of an individual coalition.")
    }
    if (j %in% 1:n == FALSE) {
      stop("'j' must be the position of an individual coalition.")
    }
  } else {
    if (codebin2lex(n=n,Nbin=i) %in% 1:n == FALSE) {
      stop("'i' must be the position of an individual coalition.")
    }
    if (codebin2lex(n=n,Nbin=j) %in% 1:n == FALSE) {
      stop("'j' must be the position of an individual coalition.")
    }
  }

  if (i == j) {
    stop("'i' and 'j' must refer to different players.")
  }

  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  # Función auxiliar 1: Subconjuntos con jugadores específicos
  subsets <- function(n, players) {
    all_coalitions <- 1:(2^n - 1)
    coalitions <- which(sapply(all_coalitions, function(x) all(players %in% which(intToBits(x)[1:n] == 1))))
    return(coalitions)
  }

  # Función auxiliar 2: Eliminación de un jugador del conjunto S
  remove_player <- function(S, i, n) {
    bits <- intToBits(S)[1:n]
    bits[i] <- as.raw(0)
    return(sum(2^(which(bits == 1) - 1)))
  }

  # Subconjuntos donde 'i' y 'j' están presentes
  idx <- subsets(n, c(i, j))

  # Verificación de la condición de simetría
  for (S in idx) {
    # Se comprueba si v(S\{i})=v(S\{j})
    S_i <- remove_player(S, i, n)  # Eliminar jugador i
    S_j <- remove_player(S, j, n)  # Eliminar jugador j
    if (abs(v[S_i]-v[S_j]) > tol) {
      return(FALSE)  # Si no se cumple la condición, no son jugadores simétricos
    }
  }

  return(TRUE)  # Si se cumple para todas las coaliciones, son jugadores simétricos
}
