#' The Unit-Garima distribution (UGa)
#'
#' @description
#' Density, distribution function, quantile function, and random generation function
#' for UGa distribution with one parameter (\code{theta}). See details in references.
#'
#' @param x,q     vector of quantile.
#' @param p       vector of probabilities.
#' @param n       number of observations.
#' @param theta   shape parameter, where \eqn{\theta > 0}.
#' @param log,log.p   logical; (default = \code{FALSE}), if \code{TRUE}, then probabilities are given as \code{log(p)}.
#' @param lower.tail  logical; if \code{TRUE} (default), probabilities are \eqn{P[X \le x]}, otherwise, \eqn{P[X > x]}.
#'
#' @import stats
#' @import lamW
#'
#' @references Ayuyuen, S., & Bodhisuwan, W. (2024). A generating family of unit-Garima distribution: Properties, likelihood inference, and application.
#' Pakistan Journal of Statistics and Operation Research, 20(1), 69-84. doi:10.18187/pjsor.v20i1.4307.
#'
#' @returns
#' \code{dUGa} gives the density,
#' \code{pUGa} gives the distribution function,
#' \code{qUGa} gives the quantile function,
#' and \code{rUGa} generates random samples.
#'
#' @export
#' @name UGa
#' @examples
#' NULL
#'

#' @export
#' @rdname UGa
#' @examples
#' x <- seq(0.1,1,by=0.1)
#' dUGa(x,theta=0.5)           #f(x)
#' dUGa(x,theta=0.5,log=TRUE)  #log(f(x))
#'
dUGa <- function(x,theta,log=FALSE){
  a<- theta*(theta+x)/((2+theta)*x^3)
  b <- exp(-theta*(1/x-1))
  fx <- a*b
  if (log==FALSE)
    return (fx)
  else
    return(log(fx))
}

#' @export
#' @rdname UGa
#' @examples
#' pUGa(x,theta=1.5)                        #P(X<x)
#' pUGa(x,theta=1.5,lower.tail = FALSE )    #P(X>x)
#'
pUGa <- function(q,theta,lower.tail=TRUE,log.p=FALSE){
  a <-  1 + (theta/(2+theta)*(1/q-1))
  b <- exp(-theta*(1/q-1))
  cdf<- a*b
  if(lower.tail==TRUE)
    p <- cdf
  else
    p <- 1-cdf

  if (log.p==TRUE)
    return (log(p))
  else
    return(p)
}

#' @export
#' @rdname UGa
#' @examples
#' # library(lamW) is required for qUGa() function
#' x <- seq(0.1,1,by=0.1)
#' x
#' p <- pUGa(x,theta=2.5)
#' p
#' require(lamW)
#' q <- qUGa(p,theta=2.5)
#' q     # q equal to x
#'
qUGa <- function(p,theta =0.5){
  u <- p
  z <- -lamW::lambertWm1(-u*(theta+2)*exp(-(theta+2)))
  x <- log(z)
  y <- -log(u*(2+theta))
  p <- 1+(1/theta)*(x+y)
  q <- 1/p
  return(q)
}

#' @export
#' @rdname UGa
#' @examples
#' # library(lamW) is required for rUGa() function
#' require(lamW)
#' x <- rUGa(100,theta=1)
#' x     #  0<x<1, for all x
rUGa <- function(n,theta){
  u <- stats::runif(n)
  z <- -lamW::lambertWm1(-u*(theta+2)*exp(-(theta+2)))
  x <- log(z)
  y <- -log(u*(2+theta))
  p <- 1+(1/theta)*(x+y)
  r <- 1/p
  return(r)
}

