% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ellcopest.R
\name{ellcopest}
\alias{ellcopest}
\title{ellcopest}
\usage{
ellcopest(
  dataU,
  Sigma_m1,
  h,
  grid,
  niter = 10,
  a,
  Kernel = "epanechnikov",
  shrink,
  verbose = 1,
  startPoint = "identity",
  prenormalization = FALSE,
  normalize = 1
)
}
\arguments{
\item{dataU}{The (estimated) copula observations from a \eqn{q}-dimensional random vector \eqn{\mathbf{X}} (\eqn{n \times q} matrix with observations in rows, variables in columns).}

\item{Sigma_m1}{The (estimated) inverse of the scale matrix of the meta-elliptical copula.}

\item{h}{The bandwidth of the kernel.}

\item{grid}{The grid of values on which to estimate the density generator.}

\item{niter}{The number of iterations used in the MECIP (default = 10).}

\item{a}{The tuning parameter to improve the performance at \eqn{0}.}

\item{Kernel}{The kernel used for the smoothing (default = "epanechnikov").}

\item{shrink}{The shrinkage function to further improve the performance at \eqn{0} and guarantee the existence of the AMISE bandwidth.}

\item{verbose}{See the \dQuote{EllDistrEst.R} function of the R package \sQuote{ElliptCopulas}.}

\item{startPoint}{See the \dQuote{EllDistrEst.R} function of the R package \sQuote{ElliptCopulas}.}

\item{prenormalization}{See the \dQuote{EllDistrEst.R} function of the R package \sQuote{ElliptCopulas}.}

\item{normalize}{A value in \eqn{\{1,2\}} indicating the normalization procedure that is applied to the estimated generator (default = 1).}
}
\value{
The estimates for \eqn{g_{\mathcal{R}}} at the grid points.
}
\description{
This functions performs improved kernel density estimation of the generator of a meta-elliptical
copula by using Liebscher's algorithm, combined with a shrinkage function.
}
\details{
The context is the one of a \eqn{q}-dimensional random vector \eqn{\mathbf{X} = (\mathbf{X}_{1}, \dots, \mathbf{X}_{k})},

with \eqn{\mathbf{X}_{i} = (X_{i1}, \dots, X_{id_{i}})} for \eqn{i = 1, \dots, k}, having a meta-elliptical copula.
This means that there exists a generator \eqn{g_{\mathcal{R}} : (0,\infty) \rightarrow \mathbb{R}} and a quantile function \eqn{Q}, such that the random vector \eqn{\mathbf{Z} = (\mathbf{Z}_{1}, \dots, \mathbf{Z}_{k})} with
\deqn{\mathbf{Z}_{i} = (Z_{i1}, \dots, Z_{id_{i}}) = \left(\left (Q \circ F_{i1} \right ) \left (X_{i1} \right ), \dots, \left (Q \circ F_{id_{i}} \right ) \left (X_{id_{i}} \right )  \right )} for \eqn{i = 1, \dots, k},
where \eqn{F_{ij}} is the cdf of \eqn{X_{ij}}, has a multivariate elliptical distribution.
Denoting \eqn{\widehat{F}_{ij}(x_{ij}) = \frac{1}{n+1} \sum_{\ell = 1}^{n} 1 \left (X_{ij}^{(\ell)} \leq x_{ij} \right )} for the (rescaled) empirical cdf of \eqn{X_{ij}} based on a sample \eqn{X_{ij}^{(1)}, \dots, X_{ij}^{(n)}} for \eqn{i = 1, \dots, k} and \eqn{j = 1, \dots, d_{i}},
and \eqn{\widehat{\mathbf{R}}} for an estimator of the scale matrix \eqn{\mathbf{R}}, this function estimates \eqn{g_{\mathcal{R}}} by using the MECIP (Meta-Elliptical Copula Iterative Procedure) of Derumigny & Fermanian (2022).

This means that we start from an initial guess \eqn{\widehat{g}_{\mathcal{R},0}} for the generator \eqn{g_{\mathcal{R}}}, based on which we obtain an estimated
sample from \eqn{\mathbf{Z}} through the quantile function corresponding to \eqn{\widehat{g}_{\mathcal{R},0}}.
Based on this estimated sample, we then obtain an estimator \eqn{\widehat{g}_{\mathcal{R},1}} using the function
\code{\link{elldistrest}}, performing improved kernel estimation with shrinkage function.
This procedure is repeated for a certain amount (niter) of iterations to obtain a final estimate for \eqn{g_{\mathcal{R}}}.

The estimator without the shrinkage function \eqn{\alpha} is implemented in the R package \sQuote{ElliptCopulas}.
We use this implementation and bring in the shrinkage function.

In order to make \eqn{g_{\mathcal{R}}} identifiable, an extra normalization procedure is implemented
in line with an extra constraint on \eqn{g_{\mathcal{R}}}.
When normalize = 1, this corresponds to \eqn{\mathbf{R}} being the correlation matrix of \eqn{\mathbf{Z}}.
When normalize = 2, this corresponds to the identifiability condition of Derumigny & Fermanian (2022).
}
\examples{
\donttest{
q = 4

# Sample size
n = 1000

# Grid on which to evaluate the elliptical generator
grid = seq(0.005,100,by = 0.005)

# Degrees of freedom
nu = 7

# Student-t generator with 7 degrees of freedom
g_q = ((nu/(nu-2))^(q/2))*(gamma((q+nu)/2)/(((pi*nu)^(q/2))*gamma(nu/2))) *
      ((1+(grid/(nu-2)))^(-(q+nu)/2))

# Density of squared radius
R2 = function(t,q){(gamma((q+nu)/2)/(((nu-2)^(q/2))*gamma(nu/2)*gamma(q/2))) *
                   (t^((q/2)-1)) * ((1+(t/(nu-2)))^(-(q+nu)/2))}

# Sample from 4-dimensional Student-t distribution with 7 degrees of freedom
# and identity covariance matrix
sample = ElliptCopulas::EllDistrSim(n,q,diag(q),density_R2 = function(t){R2(t,q)})

# Copula pseudo-observations
pseudos = matrix(0,n,q)
for(j in 1:q){pseudos[,j] = (n/(n+1)) * ecdf(sample[,j])(sample[,j])}

# Shrinkage function
shrinkage = function(t,p){1-(1/((t^p) + 1))}

# Tuning parameter selection
opt_parameters = elliptselect(n,q,seq((3/4)-(1/q)+0.01,1-0.01,len = 200),
                                  seq(0.01,2,len = 200))

# Optimal tuning parameters
a = opt_parameters$Opta ; p = opt_parameters$Optp ; h = opt_parameters$Opth

# Estimated elliptical generator
g_est = ellcopest(dataU = pseudos,Sigma_m1 = diag(q),h = h,grid = grid,a = a,
                  shrink = function(t){shrinkage(t,p)})

plot(grid,g_est,type = "l", xlim = c(0,8))
lines(grid,g_q,col = "green")
}

}
\references{
Derumigny, A., Fermanian, J.-D., Ryan, V., van der Spek, R. (2024).
ElliptCopulas, R package version 0.1.4.1.
url: https://CRAN.R-project.org/package=ElliptCopulas.

Derumigny, A. & Fermanian, J.-D. (2022).
Identifiability and estimation of meta-elliptical copula generators.
Journal of Multivariate Analysis 190:104962. \cr
doi: https://doi.org/10.1016/j.jmva.2022.104962.

De Keyser, S. & Gijbels, I. (2024).
Hierarchical variable clustering via copula-based divergence measures between random vectors.
International Journal of Approximate Reasoning 165:109090.
doi: https://doi.org/10.1016/j.ijar.2023.109090.

Liebscher, E. (2005).
A semiparametric density estimator based on elliptical distributions.
Journal of Multivariate Analysis 92(1):205-225.
doi: https://doi.org/10.1016/j.jmva.2003.09.007.
}
\seealso{
\code{\link{elldistrest}} for improved kernel estimation of the elliptical generator of an elliptical distribution,
\code{\link{elliptselect}} for selecting optimal tuning parameters for the improved kernel estimator of the elliptical generator,
\code{\link{phiellip}} for estimating the \eqn{\Phi}-dependence between \eqn{k} random vectors having a meta-elliptical copula.
}
