#' Fit Modified Weibull-Type Distributions
#'
#' @importFrom stats optim runif
#' @param x Numeric vector of positive data
#' @param dist Character; choose: "Rangoli2023","Peng2014","Lai2003",
#' "Xie1996","Sarhan2009","Rangoli2025","Mustafa2012","Alwasel2009"
#' @return List with estimates, log-likelihood, AIC, BIC
#' @export

WeibullModiAMR <- function(x, dist = "Alwasel2009"){
  x <- x[x > 0]
  dist <- switch(dist,
                 "Rangoli2023" = fit_rangoli2023(x),
                 "Peng2014"    = fit_peng(x),
                 "Lai2003"     = fit_lai(x),
                 "Xie1996"     = fit_xie(x),
                 "Sarhan2009"  = fit_sarhan(x),
                 "Rangoli2025" = fit_rangoli2(x),
                 "Mustafa2012" = fit_mustafa(x),
                 "Alwasel2009" = fit_alwasel(x),
                 stop("Distribution not supported!")
  )
}


# 1. Rangoli et al. 2023
fx_rangoli2023 <- function(x, a, b, g) {
  if(a <= 0 || b <= 0 || g <= 0 || any(x <= 0)) return(rep(1e-10, length(x)))
  s <- a * x^b - g / x
  s <- pmin(pmax(s, -700), 700)
  dens <- (a * b * x^(b - 1) + g / (x^2)) * exp(s) * exp(-exp(s))
  dens[!is.finite(dens) | dens <= 1e-20] <- 1e-10
  return(dens)
}

loglik_rangoli2023 <- function(params, data) {
  dens <- fx_rangoli2023(data, params[1], params[2], params[3])
  -sum(log(dens))
}

init_rangoli2023 <- function(x) { runif(3, 0.1, 1.5) }

fit_rangoli2023 <- function(x, max_iter = 200, tol = 1e-6) {
  params <- init_rangoli2023(x)
  loglik_old <- Inf
  for(i in 1:max_iter) {
    opt <- optim(par = params,
                 fn = loglik_rangoli2023,
                 data = x,
                 method = "L-BFGS-B",
                 lower = rep(1e-6, 3))
    loglik_new <- -opt$value
    if(abs(loglik_new - loglik_old) < tol) break
    params <- opt$par
    loglik_old <- loglik_new
  }
  n <- length(x); k <- 3
  list(estimates = opt$par,
       loglik = loglik_new,
       AIC = 2*k - 2*loglik_new,
       BIC = k*log(n) - 2*loglik_new,
       converged = opt$convergence == 0)
}


# 2. Peng & Yun 2014
fx_peng <- function(x, alpha, beta, lambda){
  term1 <- alpha * exp(-lambda / x)
  term2 <- beta * x^(beta - 1) + lambda * x^(beta - 2)
  term3 <- exp(-alpha * x^beta * exp(-lambda / x))
  res <- term1 * term2 * term3
  res[!is.finite(res) | res <= 1e-12] <- 1e-12
  res
}

loglik_peng <- function(x, alpha, beta, lambda){ sum(log(fx_peng(x, alpha, beta, lambda))) }
init_peng <- function(x){ runif(3, 0.5, 1.5) }

fit_peng <- function(x, max_iter = 100, tol = 1e-6){
  params <- init_peng(x)
  for(i in 1:max_iter){
    opt <- optim(par=params,
                 fn=function(p) -loglik_peng(x,p[1],p[2],p[3]),
                 method="L-BFGS-B",
                 lower=rep(1e-6,3))
    if(abs(-opt$value - loglik_peng(x,opt$par[1],opt$par[2],opt$par[3]))<tol) break
    params <- opt$par
  }
  list(estimates=opt$par,
       loglik=-opt$value,
       AIC=2*3-2*(-opt$value),
       BIC=3*log(length(x))-2*(-opt$value),
       converged=opt$convergence==0)
}


# 3. Lai et al. 2003
fx_lai <- function(x, a, b, l){
  term1 <- a*(b + l*x)*x^(b-1)*exp(l*x)
  term2 <- exp(-a*x^b*exp(l*x))
  res <- term1*term2
  res[!is.finite(res) | res<=1e-12] <- 1e-12
  res
}

loglik_lai <- function(x, a,b,l){ sum(log(fx_lai(x,a,b,l))) }
init_lai <- function(x){ runif(3,0.5,1.5) }

fit_lai <- function(x, max_iter=100, tol=1e-6){
  params <- init_lai(x)
  for(i in 1:max_iter){
    opt <- optim(par=params,
                 fn=function(p) -loglik_lai(x,p[1],p[2],p[3]),
                 method="L-BFGS-B",
                 lower=rep(1e-6,3))
    if(abs(-opt$value - loglik_lai(x,opt$par[1],opt$par[2],opt$par[3]))<tol) break
    params <- opt$par
  }
  list(estimates=opt$par,
       loglik=-opt$value,
       AIC=2*3-2*(-opt$value),
       BIC=3*log(length(x))-2*(-opt$value),
       converged=opt$convergence==0)
}


# 4. Xie & Lai 1996
fx_xie <- function(x, g,l,a,b){
  term1 <- a*b*x^(b-1)
  term2 <- g*l*x^(l-1)
  exp_term <- exp(-(a*x^b + g*x^l))
  res <- (term1 + term2) * exp_term
  res[!is.finite(res) | res <= 1e-12] <- 1e-12
  res
}

loglik_xie <- function(x,g,l,a,b){ sum(log(fx_xie(x,g,l,a,b))) }
init_xie <- function(x){ runif(4,0.5,1.5) }

fit_xie <- function(x,max_iter=100,tol=1e-6){
  params <- init_xie(x)
  for(i in 1:max_iter){
    opt <- optim(par=params,
                 fn=function(p) -loglik_xie(x,p[1],p[2],p[3],p[4]),
                 method="L-BFGS-B",
                 lower=rep(1e-6,4))
    if(abs(-opt$value - loglik_xie(x,opt$par[1],opt$par[2],opt$par[3],opt$par[4]))<tol) break
    params <- opt$par
  }
  list(estimates=opt$par,
       loglik=-opt$value,
       AIC=2*4-2*(-opt$value),
       BIC=4*log(length(x))-2*(-opt$value),
       converged=opt$convergence==0)
}


# 5. Sarhan & Zaindin 2009
fx_sarhan <- function(x,g,a,b){
  term1 <- a*b*x^(b-1) + g
  exp_term <- exp(-(a*x^b + g*x))
  res <- term1*exp_term
  res[!is.finite(res) | res<=1e-12] <- 1e-12
  res
}

loglik_sarhan <- function(x,g,a,b){ sum(log(fx_sarhan(x,g,a,b))) }
init_sarhan <- function(x){ runif(3,0.5,1.5) }

fit_sarhan <- function(x,max_iter=100,tol=1e-6){
  params <- init_sarhan(x)
  for(i in 1:max_iter){
    opt <- optim(par=params,
                 fn=function(p) -loglik_sarhan(x,p[1],p[2],p[3]),
                 method="L-BFGS-B",
                 lower=rep(1e-6,3))
    if(abs(-opt$value - loglik_sarhan(x,opt$par[1],opt$par[2],opt$par[3]))<tol) break
    params <- opt$par
  }
  list(estimates=opt$par,
       loglik=-opt$value,
       AIC=2*3-2*(-opt$value),
       BIC=3*log(length(x))-2*(-opt$value),
       converged=opt$convergence==0)
}


# 6. Rangoli & Talawar 2025
fx_rangoli2 <- function(x,g,l,a,b){
  term1 <- g*l*x^(l-1)
  term2 <- (a*b*x^(b-1)*exp(x^b))/b^a
  exp_term <- exp(-((a/b^a)*(exp(x^b)-1) + g*x^l))
  res <- (term1 + term2) * exp_term
  res[!is.finite(res) | res <= 1e-12] <- 1e-12
  res
}

loglik_rangoli2 <- function(x,g,l,a,b){ sum(log(fx_rangoli2(x,g,l,a,b))) }
init_rangoli2 <- function(x){ runif(4,0.5,1.5) }

fit_rangoli2 <- function(x,max_iter=100,tol=1e-6){
  params <- init_rangoli2(x)
  for(i in 1:max_iter){
    opt <- optim(par=params,
                 fn=function(p) -loglik_rangoli2(x,p[1],p[2],p[3],p[4]),
                 method="L-BFGS-B",
                 lower=rep(1e-6,4))
    if(abs(-opt$value - loglik_rangoli2(x,opt$par[1],opt$par[2],opt$par[3],opt$par[4]))<tol) break
    params <- opt$par
  }
  list(estimates=opt$par,
       loglik=-opt$value,
       AIC=2*4-2*(-opt$value),
       BIC=4*log(length(x))-2*(-opt$value),
       converged=opt$convergence==0)
}


# 7. Mustafa et al. 2012
fx_mustafa <- function(x, alpha, beta, lambda, theta){
  term1 <- (alpha * lambda * theta * x^(theta - 1)) * (1 + lambda * x^theta)^(alpha - 1) + beta
  term2 <- exp(1 - (1 + lambda * x^theta)^alpha - beta * x)
  res <- term1 * term2
  res[!is.finite(res) | res <= 1e-12] <- 1e-12
  return(res)
}

loglik_mustafa <- function(x, alpha, beta, lambda, theta){
  sum(log(fx_mustafa(x, alpha, beta, lambda, theta)))
}

init_mustafa <- function(x){ runif(4, 0.5, 1.5) }

fit_mustafa <- function(x, max_iter = 100, tol = 1e-6){
  params <- init_mustafa(x)
  for(i in 1:max_iter){
    opt <- optim(par = params,
                 fn = function(p) -loglik_mustafa(x, p[1], p[2], p[3], p[4]),
                 method = "L-BFGS-B",
                 lower = rep(1e-6, 4))
    if(abs(-opt$value - loglik_mustafa(x, opt$par[1], opt$par[2], opt$par[3], opt$par[4])) < tol) break
    params <- opt$par
  }
  list(estimates = opt$par,
       loglik = -opt$value,
       AIC = 2*4 - 2*(-opt$value),
       BIC = 4*log(length(x)) - 2*(-opt$value),
       converged = opt$convergence == 0)
}


# 8. Alwasel 2009
fx_alwasel <- function(x,a,b,g){
  term1 <- a + b*g*x^(g-1)
  term2 <- exp(-(a*x + b*x^g))
  res <- term1 * term2
  res[!is.finite(res) | res <= 1e-12] <- 1e-12
  res
}

loglik_alwasel <- function(x,a,b,g){ sum(log(fx_alwasel(x,a,b,g))) }
init_alwasel <- function(x){ runif(3,0.5,1.5) }

fit_alwasel <- function(x,max_iter=100,tol=1e-6){
  params <- init_alwasel(x)
  for(i in 1:max_iter){
    opt <- optim(par=params,
                 fn=function(p) -loglik_alwasel(x,p[1],p[2],p[3]),
                 method="L-BFGS-B",
                 lower=rep(1e-6,3))
    if(abs(-opt$value - loglik_alwasel(x,opt$par[1],opt$par[2],opt$par[3]))<tol) break
    params <- opt$par
  }
  list(estimates=opt$par,
       loglik=-opt$value,
       AIC=2*3-2*(-opt$value),
       BIC=3*log(length(x))-2*(-opt$value),
       converged=opt$convergence==0)
}
