#################################################
# Main function for new entropy estimation procedure.
# Originally from: entEst_newSepFNNUNdynamic.R
# Dependencies: entropyWts.R, FNN package
#################################################

#' @title New Entropy Estimator
#' @description Entropy estimator based on k-nearest neighbors.
#' @param summaries data matrix with summary statistics as rows and samples as columns (d x m)
#' @param k user-chosen k, or NULL
#' @param terms user-supplied terms, or NULL
#' @param wts user-supplied weights, or NULL
#' @return Estimated differential entropy.
#' @examples
#' x=matrix(rnorm(2500),nrow=100)
#' entropy.true=.5*25*log(2*pi*exp(1))
#' entropy=entEst(t(x))
#' @export
entEst <- function(
    summaries,
    k     = NULL,    # user-chosen k, or NULL
    terms = NULL,    # user-supplied terms, or NULL
    wts   = NULL     # user-supplied weights, or NULL
) {
  # 'summaries' is d x m
  d <- nrow(summaries)
  m <- ncol(summaries)
  #a=log(m)/log(k)
  # if terms or wts is NULL, compute them:
  if (is.null(terms) || is.null(wts)) {
    tmp   <- entropyWts(d = d, m = m,
                        k = k, option = 3)  # pre-defined function
    terms <- tmp$terms
    wts   <- tmp$wts
  }

  # 1) need up to 'kmax = max(terms)' nearest neighbors
  kmax <- max(terms)

  # 2) transpose the data for FNN: m observations of dimension d
  X <- t(summaries)  # shape: m x d

  # 3) distances to the 'kmax' nearest neighbors (excluding self)
  nnDist <- FNN::knn.dist(X, k = kmax, algorithm = "kd_tree")
  # nnDist is m x kmax; row i = distances to nearest neighbors for sample i

  # 4) only want the columns in 'terms' => j-th nearest neighbors
  #    'med' becomes an m x length(terms) matrix
  med <- nnDist[, terms, drop = FALSE]
  # print(med)

  # 5) compute the final estimate per eqn (15)
  vd <- log((m - 1) * pi^(d / 2)) - lgamma(1 + 0.5 * d)
  cn <- sum(wts * digamma(terms))/sum(wts)
  ad <- d * mean(log(med) %*% wts)/sum(wts)

  estEnt <- vd - cn + ad
  return(estEnt)
}
