test_that("sim_list construction works as expected", {
  config <- create_locations(20, 30, seed = 120)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  model_list <- as.list(models)
  models_2 <- sim_list(model_list)
  expect_equal(models, models_2)
  ## error detection
  expect_error(models[[20]])
  expect_error(sim_list(12))
  expect_error(sim_list(list(model_list[[1]], "foo")))
  model_list_bug <- model_list
  model_list_bug[[2]]$costs <- model_list_bug[[2]]$costs + 1
  expect_error(sim_list(model_list_bug))
  model_list_bug <- model_list
  origin_names(model_list_bug[[2]]) <- letters[1:20]
  expect_error(sim_list(model_list_bug))
  model_list_bug <- model_list
  destination_positions(model_list_bug[[2]]) <- config$pd
  expect_error(sim_list(model_list_bug))
})

test_that("sim_list is a list", {
  config <- create_locations(20, 30, seed = 4)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    epsilon = 0.1,
    iter_max = 5000,
    precision = .Machine$double.eps^0.5
  )
  expect_length(models, length(alphas) * length(betas))
  expect_s3_class(models, "sim_list")
  for (k in seq_along(models)) {
    expect_s3_class(models[k], "sim_list")
    expect_s3_class(models[[k]], "sim_blvim")
    expect_mapequal(unclass(models[k][[1]]), unclass(models[[k]]))
  }
  params <- expand.grid(alpha = alphas, beta = betas)
  expect_equal(params$alpha, sapply(models, return_to_scale))
  expect_equal(params$beta, sapply(models, inverse_cost))
  expect_equal(config$costs, costs(models))
})

test_that("sim_list prints as expected", {
  config <- create_locations(20, 30, seed = 5)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    epsilon = 0.1,
    iter_max = 5000,
    precision = .Machine$double.eps^0.5
  )
  expect_snapshot(models)
})

test_that("sim_list attractivenesses extraction works", {
  config <- create_locations(20, 30, seed = 4)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    epsilon = 0.1,
    iter_max = 5000,
    precision = .Machine$double.eps^0.5
  )
  grid_Z <- grid_attractiveness(models)
  expect_true(inherits(grid_Z, "matrix"))
  expect_equal(dim(grid_Z), c(length(alphas) * length(betas), length(config$Z)))
  for (k in seq_along(models)) {
    expect_equal(grid_Z[k, ], attractiveness(models[[k]]))
  }
})

test_that("sim_list destination flows extraction works", {
  config <- create_locations(20, 30, seed = 4)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    epsilon = 0.1,
    iter_max = 5000,
    precision = .Machine$double.eps^0.5
  )
  grid_Z <- grid_destination_flow(models)
  expect_true(inherits(grid_Z, "matrix"))
  expect_equal(dim(grid_Z), c(length(alphas) * length(betas), length(config$Z)))
  for (k in seq_along(models)) {
    expect_equal(grid_Z[k, ], destination_flow(models[[k]]))
  }
})

test_that("sim_list is_terminal extraction works", {
  config <- create_locations(20, 20, seed = 42, symmetric = TRUE)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    bipartite = FALSE,
    epsilon = 0.1,
    iter_max = 5000,
    precision = .Machine$double.eps^0.5
  )
  grid_term <- grid_is_terminal(models)
  expect_true(inherits(grid_term, "matrix"))
  expect_equal(dim(grid_term), c(length(alphas) * length(betas), length(config$Z)))
  for (k in seq_along(models)) {
    expect_equal(grid_term[k, ], is_terminal(models[[k]]))
  }
})

test_that("sim_list grid_* redundant functions work as expected", {
  config <- create_locations(20, 30, seed = 4)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    epsilon = 0.1,
    iter_max = 400,
    precision = .Machine$double.eps^0.5
  )
  expect_equal(grid_sim_converged(models), sim_converged(models))
  expect_equal(grid_sim_iterations(models), sim_iterations(models))
  expect_equal(grid_diversity(models), diversity(models))
})

test_that("sim_list extraction functions detect errors", {
  expect_error(grid_destination_flow(list()))
  expect_error(grid_attractiveness(2))
  expect_error(grid_is_terminal(data.frame(x = 1:10)))
})

test_that("sim_list common information restoration works", {
  config <- create_locations(20, 30, seed = 20)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  ## introduction names
  on <- paste(sample(letters, 20, replace = TRUE), 1:20, sep = "_")
  dn <- paste(sample(LETTERS, 30, replace = TRUE), 1:30, sep = "_")
  rownames(config$costs) <- on
  colnames(config$costs) <- dn
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  expect_equal(costs(models), config$costs)
  for (k in seq_along(models)) {
    expect_equal(origin_names(models[[k]]), on)
    expect_equal(destination_names(models[[k]]), dn)
    expect_equal(costs(models[[k]]), config$costs)
  }
})

test_that("sim_list modifications", {
  config <- create_locations(20, 30, seed = 120)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  save_one <- models[[1]]
  models[[1]] <- models[[2]]
  expect_identical(models[[1]], models[[2]])
  models[1:3] <- models[5:7]
  for (k in 1:3) {
    expect_identical(models[[k]], models[[k + 4]])
  }
  ## error cases
  expect_error(models[[c(1, 2)]] <- list(save_one, save_one))
  expect_error(models[[1]] <- models)
  expect_error(models[[20]] <- save_one)
  expect_error(models[1] <- models[[1]])
  expect_error(models[c(1, 20)] <- sim_list(list(save_one, save_one)))
  wrong_model <- save_one
  destination_names(wrong_model) <- sample(letters, 30, replace = TRUE)
  expect_error(models[[1]] <- wrong_model)
  wrong_model <- save_one
  origin_names(wrong_model) <- sample(letters, 20, replace = TRUE)
  expect_error(models[[1]] <- wrong_model)
  config2 <- create_locations(10, 20, seed = 0)
  models2 <- grid_blvim(config2$costs,
    config2$X,
    alphas,
    betas,
    config2$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  expect_error(models[[1]] <- models2[[1]])
  expect_error(models[1:3] <- models2[1:3])
})

test_that("sim_list out of range access tentative are detected", {
  config <- create_locations(20, 30, seed = 480)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  expect_error(models[c(1, 5, 20)], regexp = "out of range value")
  expect_error(models[[14:18]], regexp = "more than one index")
})

test_that("sim_list c concatenates sim_lists correctly", {
  config <- create_locations(20, 30, seed = 0)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  res <- c(models, models)
  expect_length(res, 2L * length(models))
  for (k in seq_len(length(models))) {
    expect_identical(res[[k]], models[[k]])
    expect_identical(res[[k + length(models)]], models[[k]])
  }
  expect_identical(res[seq_along(models)], models)
  expect_identical(res[length(models) + seq_along(models)], models)
  a_model <- blvim(config$costs,
    config$X,
    1.1,
    5,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  res2 <- c(models, a_model)
  expect_length(res2, 1L + length(models))
  for (k in seq_len(length(models))) {
    expect_identical(res2[[k]], models[[k]])
  }
  expect_mapequal(unclass(res2[[length(models) + 1]]), unclass(a_model))
})

test_that("sim_list c detects errors", {
  config <- create_locations(20, 30, seed = 0)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  expect_error(c(models, 1L))
  config2 <- create_locations(10, 20, seed = 0)
  models2 <- grid_blvim(config2$costs,
    config2$X,
    alphas,
    betas,
    config2$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  expect_error(c(models, models2))
  expect_error(c(models, models2[[1]]))
  models3 <- models
  origin_names(models3) <- sample(letters, 20)
  expect_error(c(models, models3))
  models3 <- models
  destination_names(models3) <- sample(letters, 30, replace = TRUE)
  expect_error(c(models, models3))
})

test_that("sim_list conversion to data frame", {
  config <- create_locations(20, 30, seed = 120)
  alphas <- seq(1.25, 2, by = 0.25)
  betas <- 1 / seq(0.1, 0.5, length.out = 4)
  models <- grid_blvim(config$costs,
    config$X,
    alphas,
    betas,
    config$Z,
    iter_max = 5000,
    epsilon = 0.1,
    precision = .Machine$double.eps^0.5
  )
  models_df <- as.data.frame(models, sim_column = "aname")
  expect_s3_class(models_df, "data.frame")
  expect_equal(dim(models_df), c(length(models), 1L))
  expect_named(models_df, "aname")
  ## we need the I
  expect_identical(models_df$aname, I(models))
  ## error case
  expect_error(as.data.frame(models, sim_column = 1:2))
})
