## 
# ------------------------------------------------------------------------
# 
# "bootsemi(x,func,B,...,model,params,model.fit,model.sim,H0,PAR0)" --
# 
# Bootstrap for time series.
# 
# ------------------------------------------------------------------------
##
##
#' @aliases bootsemi
#' @title Semiparametric Bootstrap
#' @description The function performs a semiparametric bootstrap for a general
#' statistics, using a time-series model.
#' @param x A vector or a time series representing the data.
#' @param func The function to apply to each sample.
#' @param B A positive integer; the number of bootstrap replications.
#' @param ... Optional additional arguments for the \code{func} function.
#' @param model The chosen model to fit the time series. Either "ARIMA" or "GARCH", else
#' \code{model.fit} and \code{model.sim} should be specified.
#' @param params The parameters of the model (see below).
#' @param model.fit A function fitting the parameters for the generic model (see below).
#'  By default \code{NULL}.
#' @param model.sim A function simulating the data for the generic model (see below). By default
#' \code{NULL}.
#' @param H0 Logical. Only implemented for the ARIMA and GARCH models.
#'  If \code{H0 = TRUE}, the value(s) of the parameter(s) under the null
#'  \code{PAR0} should be specified. Then, the model is simulated
#'  under the null hypothesis. By default it is \code{NULL}.
#' @param PAR0 The value of the parameter that we want to test under the null hypothesis.
#'  By default it is \code{FALSE}.
#' @details The default basic models currently supported are : \emph{ARIMA} and \emph{GARCH}.
#' 
#' The argument \code{params} specifies the orders of the chosen model. In
#' the case of \emph{ARIMA}, it is a vector of the form \emph{c(p,q)} or
#' \emph{c(p,d,q)}. In the case of \emph{GARCH}, it is a vector of the form
#' \emph{c(q)} or \emph{c(p,q)} corresponding to an \emph{ARCH(q)} or 
#' \emph{GARCH(p,q)} model, respectively.
#' 
#' Alternatively, one can specify two functions in the \code{model.fit}  and
#' \code{model.sim} arguments. They are used to implement a generic bootstrap procedure, which 
#' first estimates the model with \code{model.fit} and then simulates the bootstrap time series
#' with \code{model.sim}. The 
#' \code{model.fit} function has the following prototype:
#'   \preformatted{
#'     model.fit(x,params)
#'   }
#' It receives the \code{params} argument specified in the \code{bootsemi} function.
#' It should return an object describing the model (typically a list
#' containing all the necessary components for the model). The
#' \code{model.sim} function has the following prototype:
#'   \preformatted{
#'     model.sim(model,innovations,params)
#'   }
#' The \code{innovations} argument is the resampled vector of centered residuals.
#' The function builds a new trajectory of the original process using the
#' estimators provided by the \code{model.fit} function.
#' 
#' The Examples section below shows how this can be done in the case of a
#' Threshold Autoregressive (\emph{TAR}) process.
#' 
#' Parameters \code{HO} and \code{PAR0} may be used to generate the bootstrap distribution 
#' of the statistic of interest under a specific null hypothesis. This option may be
#' useful, for instance, for ARIMA models with unit roots. In that case, the usual bootstrap
#' dose not work, unless one simulates the process under the null hypothesis of unit root.
#' 
#' @return \code{bootsemi} returns an object of class \code{boodd} 
#' (see \link{class.boodd}).
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Lahiri, S. N. (2003). \emph{Resampling Methods for Dependent Data}. Springer N.Y.. 
#' 
#' Shimizu, K. .(2017). \emph{Bootstrapping Stationary ARMA-GARCH Models}.
#'  Springer Fachmedien Wiesbaden. 
#'  
#' Park, J.Y. (2003). Bootstrap unit-root tests, \emph{Econometrica}, \bold{77}, 1845-1895.
#'  
#' @seealso
#' \code{\link{blockboot}},
#' \code{\link{plot.boodd}},
#' \code{\link{confint.boodd}}.
#' @keywords "Semiparametric bootstrap" "ARMA" "GARCH" "TAR" "Bootstrap under the null"
#' @export
#' @examples 
#' \donttest{
#' # An ARIMA(2,1) process
#' library(stats)
#' B <- 299
#' n <- 200
#' x <- arima.sim(model=list(ar=c(0.8,-0.4),ma=c(0.2)),n=n)
#' boo1 <- bootsemi(x,mean,B,model="ARIMA",params=c(2,1))
#' plot(boo1)
#' }


bootsemi <- function(x,func,B,model=c("ARIMA","GARCH"),params,model.fit=NULL,model.sim=NULL,H0=FALSE,PAR0=NULL,...) {
  # Test the value returned by func
  func_args <- names(formals(func))
  if ("params" %in% func_args) {
    y <- func(x, params, ...)
  } else {
    y <- func(x, ...)
  }
  if (!is.vector(y)) {
    stop("Function 'func' must return a vector")
  }
  len <- length(y)
  cnames <- names(y)
  
  if (!is.null(model.fit) || !is.null(model.sim)) {
    if (!is.null(model.fit) && !is.null(model.sim)) {
      res <- bootsemi.generic(x,func,B,model.fit,model.sim,params,...)
    } else {
      stop("both arguments 'model.fit' and 'model.sim' are required.")
    }
  } else {
    model <- match.arg(model)
    if (model[1] == "ARIMA") {
      res <- bootsemi.arima(x,func,B,params,H0,PAR0,...)
    } else if (model[1] == "GARCH") {
      res <- bootsemi.garch(x,func,B,params,H0,PAR0,...)
    } else {
      stop("unknown model ",model[1])
    }
  }
  
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  res=as.matrix(res)
  if (H0==FALSE) obj <- list(s=res,Tn=y) else obj <- list(s=res,Tn=colMeans(res))
  class(obj) <- "boodd"
  attr(obj,"kind") <- "semi"
  attr(obj,"func") <- func
  return(obj)
}


## 
# ------------------------------------------------------------------------
# 
# "bootsemi.generic <- function(x,func,B,model.fit,model.sim,params,...)" --
# 
# Bootstrap for generically modelled time series.
# 
# ------------------------------------------------------------------------
##
bootsemi.generic <- function(x,func,B,model.fit,model.sim,params,...) {
  n <- length(x)
  func_args <- names(formals(func))
  if ("params" %in% func_args) {
    y <- func(x, params, ...)
  } else {
    y <- func(x, ...)
  }
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  
  # Fit the model
  fit <- model.fit(x,params)
  # Centered residuals epsilon-tilde
  eps_hat <- fit$residuals
  eps_tilde <- eps_hat - mean(eps_hat)
  
  for (i in 1:B) {
    # Bootstrap the centered residuals
    eps_star <- sample(eps_tilde,size=n,replace=TRUE)
    xstar <- model.sim(fit,eps_star,params)
    # Use conditional parameter passing based on whether 'func' expects 'params'
    if ("params" %in% func_args) {
      res[i,] <- func(xstar, params, ...)
    } else {
      res[i,] <- func(xstar, ...)
    }
  }
  return(res)
}


## 
# ------------------------------------------------------------------------
# 
# "bootsemi.arima <- function(x,func,B,params,H0=FALSE,PAR0=NULL,...)" --
# 
# Bootstrap for arima modelled time series. The arima() function from the 
# stats package uses the following definition of the ARMA model:
#     X[t] = a[1]X[t-1] + … + a[p]X[t-p] + e[t] + b[1]e[t-1] + … + b[q]e[t-q]
# If H0=TRUE, then PAR0, the values of the parameters under H0 should be specified 
# and in conformity with the values in params
# ------------------------------------------------------------------------
##
bootsemi.arima <- function(x,func,B,params,H0=FALSE,PAR0=NULL,...) {
  n <- length(x)
  #y <- func(x,...)
  y <- func(x)
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  lor <- length(params)
  if ((lor != 2) && (lor != 3)) {
    stop("wrong 'params' argument. Should be vector (p,q) or (p,d,q).")
  }
  if (length(params) == 2) {
    # Assume (p,q) specified and d=0
    params[3] <- params[2]
    params[2] <- 0
  }
  p <- params[1]
  d <- params[2]
  q <- params[3]
  phi <- 0
  theta <- 0
  # Fit the ARIMA model
  fit <- arima(x,params,include.mean=FALSE,method="CSS",transform.pars=FALSE)
  # Estimated coefficients
  coeffs <- coef(fit)
  if (H0==TRUE) {
    if (length(coeffs)==length(PAR0)) coeffs=PAR0
    else {stop("Parameters under the null do not have the correct size!")} 
  }
  if (p > 0) {
    phi <- coeffs[1:p] #   
  }
  if (q > 0) {
    theta <- coeffs[(p+1):(p+q)]
  }	
  
  # Centered residuals epsilon-tilde
  eps_hat <- residuals(fit)
  eps_tilde <- eps_hat - mean(eps_hat)
  xstar <- numeric(n)
  if (p > 0) {
    xstar[1:p] <- x[1:p]
  }
  
  for (i in 1:B) {
    # Bootstrap the centered residuals
    neweps <- sample(eps_tilde,size=n,replace=TRUE)
    for (t in (p+1):n) {
      xstar[t] <- neweps[t]
      if (p>0) {
        xstar[t] <- xstar[t]+sum(phi*xstar[(t-1):(t-p)])
      }
      if (q>0 && t>q) {
        xstar[t] <- xstar[t]+sum(theta*neweps[(t-1):(t-q)])
      }
    }
    res[i,] <- func(xstar,...)
  }
  return(res)
}


# tools::assertError

## 
# ------------------------------------------------------------------------
# 
# "bootsemi.garch <- function(x,func,B,params,...)" --
# 
# Residual bootstrap for GARCH modelled variance.
# 
# epsilon_t = sqrt(h_t) * eta_t = sigma_t * eta_t
# h_t = sigma_t^2 = omega + \sum _{i=1}^{q} alpha_{i} epsilon_{t-i}^{2} + \sum_{i=1}^{p} beta_{i} sigma_{t-i}^{2}
# 
# ------------------------------------------------------------------------
##
bootsemi.garch <- function(x,func,B,params=c(1,1),...) {
  n <- length(x)
  y <- func(x,...)
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  lor <- length(params)
  if ((lor != 1) && (lor != 2)) {
    stop("wrong 'params' argument. Should be vector (q) or (p,q).")
  }
  if (length(params) == 1) {
    # Assume it is an ARCH(q)
    params[2] <- params[1]
    params[1] <- 0
  }
  p <- params[1]
  q <- params[2]
  if (q < 1) {stop("One must have q>0.")}
  
  aa=c("~garch(",q,",",p,")")
  mod=as.formula(paste(aa,collapse=""))
  
  
  # Step 1: fit the GARCH model
  fit <- garchFit(formula=mod,data=x,include.mean=FALSE, trace = FALSE)
  coeffs <- fit@fit$coeff # Modification
  omega <- coeffs[1]
  alpha <- coeffs[2:(q+1)]
  beta <- coeffs[(q+2):(p+q+1)]
  
  # Changed condition here
  if (sum(alpha) + sum(beta) >= 1) {
    warning("the fitted series does not verify the stationarity condition (the sum of the coefficients is greater than 1)")
  }
  
  
  # Step 2: compute the estimated heteroscedasticity. It is returned in fit@sigma.t .  
  sigt=fit@sigma.t
  etahat <- as.vector(x)/sigt
  etatil <- scale(etahat)
  
  for (i in 1:B) {
    # Step 4: draw from eta_tilde
    etastar <- sample(etatil,n,replace=TRUE)
    
    # Step 5: generate the bootstrapped process
    epsstar <- sigt*etastar
    res[i,] <- func(epsstar,...)
  }
  
  return(res)
}