## 
# ------------------------------------------------------------------------
# 
# "fastNadaraya(x,bandwidth)" --
# 
# Compute the Nadaraya-Watson estimator p_n(X_i,X_{i+1}) of the transition
# density p(x,y).
# 
# ------------------------------------------------------------------------
##
#' @aliases fastNadaraya
#' @title Nadaraya-Watson Estimator for Transition Densities for Markov chains.
#' @description 
#' Calculates the Nadaraya-Watson estimator for estimating the transition densities 
#' of a Markov chain. This function is particularly useful for approximating 
#' transition kernels in Markov chain analysis.
#' @param x A numeric vector representing a Markov process.
#' @param bandwidth A real number; the kernel bandwidth smoothing parameter
#' @details 
#' The \code{fastNadaraya} function computes the estimated transition densities 
#' \eqn{p_n(X_i, X_{i+1})} of a Markov chain. It is based on the Nadaraya kernel-type 
#' estimator for the transition density, with a bandwidth \code{bandwidth} provided by the user. 
#' This function is used in conjunction with \code{\link{findBestEpsilon}} 
#' to determine optimal small set for the Regenerative Block Bootstrap \code{\link{regenboot}}. 
#' @return Returns a numeric vector of size \code{length(x) - 1}, containing 
#' the estimated transition densities for each state transition in the Markov chain.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P., Clémençon, S. (2006a). Regenerative Block
#' Bootstrap for Markov Chains. \emph{Bernoulli},  \bold{12}, 689-712.
#' 
#' Bertail, P. and Clémençon, S. (2006). 
#' \emph{Regeneration-based statistics for Harris recurrent
#'   Markov chains}, pages 1-54.
#' Number 187 in Lecture notes in Statistics. Springer.
#' 
#' Radulović, D. (2006). Renewal type bootstrap for Markov chains. \emph{Test}, 
#' \bold{13}, 147-192.
#' 
#' @seealso \code{\link{findBestEpsilon}},
#' \code{\link{regenboot}},
#' \code{\link{bandw1}}.
#' 
#' @keywords "Regenerative Block Bootstrap" "Markov chains" "Transition density estimation" 
#' "Kernel methods"
#' @export
#' @examples
#' x = arima.sim(1000, model=list(ar=c(0.4)))
#' h = bandw1(x)
#' fastNadaraya(x,h)
#' 
fastNadaraya <- function(x,bandwidth) {
  n <- length(x)-1
  Xi <- x[1:n]
  Xip1 <- x[2:(n+1)]
  f <- numeric(n)
  for (i in 1:n) {
    f[i] <- mean(dnorm((x[i]-Xi)/bandwidth)*dnorm((x[i+1]-Xip1)/bandwidth))/mean(dnorm((x[i]-x)/bandwidth))/bandwidth
  }
  return(f)
}