##
# ------------------------------------------------------------------------
#
# "plot.boodd <- function(x,...)" --
#
#
# ------------------------------------------------------------------------
##
#' @aliases plot.boodd
#' @title Plot an Object of Class \code{boodd}
#' @description Plots histograms or density estimates for objects of class \code{boodd}, 
#'              which are returned by bootstrap functions such as boots, 
#'              \code{\link{bootsemi}}, \code{\link{blockboot}}, \code{\link{regenboot}}, etc.
#' @param x An object of class \code{boodd}.
#' @param with.density Logical value indicating whether to plot the estimated density 
#'        of the bootstrap distribution (default is TRUE).
#' @param which Specifies which columns of the data to plot.
#' @param byrow Logical value indicating whether to display the matrix of 
#' histograms by row. By default it is \code{FALSE}.
#' @param ... Optional additional arguments for the \code{plot} function.
#' @details The function \code{plot.boodd} plots histograms or density estimates of 
#'          the output data contained in an object of class \code{boodd}. When the data have 
#'          more than one column, the function can display a matrix of (at most 6) histograms. 
#'          The argument \code{which} allows the selection of specific columns to plot, 
#'          while \code{byrow} controls the layout of the matrix display.
#' @return The function returns an invisible list containing the output of the
#'           \code{hist} function.
#'         In the case of multiple histograms, it returns a list of lists.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#'             
#'  Efron, B., Tibshirani, R. (1993). 
#'   \emph{An Introduction to the Bootstrap}, Chapman and Hall.
#'   
#' @seealso \code{\link{confint.boodd}}, \code{\link{summary.boodd}}, \code{\link{class.boodd}}.
#' @keywords "Bootstrap distribution" Plotting
#' @export
#' @examples
#' B <- 299
#' x <- round(rnorm(15),3)
#' boo1 <- boots(x, mean, B)
#' plot(boo1)
#' 
#' # Bootstrap of several statistics 
#' mv <- function(data) {c(mean(data), var(data))} # compute both mean and variance
#' boo2 <- boots(x, mv, B)
#' plot(boo2)
##

plot.boodd <- function(x,with.density=TRUE,which,byrow=FALSE,...) {
  iargs <- list(...)
  s <- x$s
  if (!is.matrix(x$s)) {
    s <- matrix(x$s,ncol=1)
    colnames(s) <- deparse(substitute(x))
  }
  nc <- ncol(s)
  if (missing(which)) {
    which <- 1:nc
  }
  if (length(which) > 6) {
    which <- which[1:6]
    warning("can't display more than 6 columns.")
  }
  len <- length(which)
  if (len > 1) {
    vals <- 1:len
    if ((len %% 2) == 1) {vals <- append(vals,len+1)}
    layout(matrix(vals,nrow=2,byrow=byrow))
  }
  res <- list()
  names <- colnames(s)
  for (i in 1:len) {
    oargs <- iargs
    ci <- which[i]
    if (ci > nc) {
      stop("column index out of bounds")
    }
    X <- s[,ci]
    oargs$x <- X
    oargs$prob <- TRUE
    cname <- names[ci]
    if (is.null(cname) || cname == "") {
      cname <- paste("column",ci)
    }
    if (length(iargs) == 0 || is.null(iargs$main) || is.na(iargs$main[i])) {
      oargs$main <- paste("Histogram of",cname)
    } else {
      oargs$main <- iargs$main[i]
    }
    if (length(iargs) != 0 && !is.null(iargs$xlab) && !is.na(iargs$xlab[i])) {
      oargs$xlab <- iargs$xlab[i]
    } else if (!is.null(cname)) {
      oargs$xlab <- cname
    } else {
      oargs$xlab <- paste("x",i,sep="")
    }
    
    if (with.density) {
      H <- hist(X,plot=FALSE)
      D <- density(X)
      yl <- c(0,max(max(H$density),max(D$y)))
      oargs$ylim <- yl
      H <- do.call(hist,oargs)
      lines(density(X))
    } else {
      H <- do.call(hist,oargs)
    }
    res[[i]] <- H
  }
  if (len == 1) {res <- res[[1]]}
  layout(1)
  invisible(res)
}