##
# ------------------------------------------------------------------------
#
# "sieveboot <- function(x,func,B,order,...)" --
#
# Autoregressive Sieve Bootstrap for stationary time series that admit a
# MA(∞) representation.
#
# ------------------------------------------------------------------------
##
#' @aliases sieveboot
#' @title Autoregressive Sieve Bootstrap
#' @description Applies autoregressive sieve bootstrap to stationary time series.
#' @param x A vector or time series.
#' @param func The function applied to each bootstrap sample.
#' @param B A  positive integer; the number of bootstrap replications.
#' @param order A positive integer; represents the order of the sieve autoregressive process.
#'              If not provided, it is automatically determined (see details below). 
#' @param ... Optional additional arguments for the \code{func} function.
#' @details The sieve bootstrap estimates an AR(\eqn{p}) 
#'          model, with a large \code{order} equal to \eqn{p}, resamples
#'              the centered estimated residuals, and reconstructs an AR(\eqn{p})
#'              bootstrap time series to compute a given statistic. 
#'          The default \code{order}, if not specified, is set to
#'           \eqn{4*(n^{1/4})/\log(n)^{1/2}}, 
#'           where \eqn{n} is the sample size.
#' @return Returns an object of class \code{boodd}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#'Bühlmann, P. (1997). Sieve Bootstrap for time series.
#'\emph{Bernoulli,} \bold{3}, 123-148.
#'              
#'Choi, E., Hall, P. (2000). Bootstrap confidence regions computed from autoregressions 
#' of arbitrary order. \emph{Journal of the Royal Statistical Society},
#' Series B, \bold{62}, 461-477.
#'              
#' @seealso  \code{\link{aidedboot}}.
#' @keywords "Sieve bootstrap" "Time series" "Autoregressive process" 
#' @export
#' @examples 
#' n <- 200
#' B <- 599
#' x <- arima.sim(list(order=c(0,0,4),ma=c(0.7,0.4,-0.3,-0.1)),n=n)
#' b1 <- sieveboot(x,mean,B,order=10)
#' plot(b1)
## 
sieveboot <- function(x,func,B,order=NULL,...) {
  x <- as.vector(x)
  n <- length(x)
  y <- func(x,...)
  len <- length(y)
  cnames <- names(y)
  res <- matrix(nrow=B,ncol=len)
  
  if (is.null(order)) {
    #ln <- floor(n^0.25/sqrt(log(n)))
    ln <- floor(4*n^0.25/log(n)^0.5)
  } else {
    ln <- order
  }
  if (n <= ln) {
    stop("length of series must be greater than order")
  }
  
  # Fit the linear autoregressive sieve process
  fit <- arima(x, order=c(ln,0,0))
  coeffs <- coef(fit)
  phi <- coeffs[1:ln]
  eps_hat <- residuals(fit)
  # Remove the first ln values and center on the mean
  eps_hat <- eps_hat[-(1:ln)]
  eps_tilde <- eps_hat - sum(eps_hat)/(n-ln)
  
  xstar <- numeric(n)
  xstar[1:ln] <- x[1:ln]
  
  for (i in 1:B) {
    # Bootstrap the centered residuals
    eps_star <- sample(eps_tilde,size=n,replace=TRUE)
    for (t in (ln+1):n) {
      xstar[t] <- sum(phi*xstar[(t-1):(t-ln)])+eps_star[t]
    }
    res[i,] <- func(xstar,...)
  }
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- "sieve"
  attr(obj,"func") <- func
  return(obj)
}