% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cellkey_pkg.R
\name{ck_class}
\alias{ck_class}
\alias{cellkey_pkg}
\alias{ck_setup}
\title{R6 Class defining statistical tables that can be perturbed}
\usage{
ck_setup(x, rkey, dims, w = NULL, countvars = NULL, numvars = NULL)
}
\arguments{
\item{x}{an object coercible to a \code{data.frame}}

\item{rkey}{either a column name within \code{x} referring to a variable containing record keys
or a single integer(ish) number > \code{5} that referns to the number of digits for record keys that
will be generated internally.}

\item{dims}{a list containing slots for each variable that should be
tabulated. Each slot consists should be created/modified using \code{\link[sdcHierarchies:hier_create]{sdcHierarchies::hier_create()}},
\code{\link[sdcHierarchies:hier_add]{sdcHierarchies::hier_add()}} and other functionality from package \code{sdcHierarchies}.}

\item{w}{(character) a scalar character referring to a variable in \code{x} holding sampling
weights. If \code{w} is \code{NULL} (the default), all weights are assumed to be \code{1}}

\item{countvars}{(character) an optional vector containing names of binary (0/1 coded)
variables withing \code{x} that should be included in the problem instance.
These variables can later be perturbed.}

\item{numvars}{(character) an optional vector of numerical variables that can later be tabulated.}
}
\value{
A new \code{cellkey_obj} object. Such objects (internally) contain the fully computed
statistical tables given input microdata (\code{x}), the hierarchical definitionals (\code{dims}) as
well as the remaining inputs. Intermediate results are stored internally and can only be
modified / accessed via the exported public methods described below.
}
\description{
This class allows to define statistical tables and perturb both
count and numerical variables.
}
\details{
Such objects are typically generated using \code{\link[=ck_setup]{ck_setup()}}.
}
\examples{
\donttest{
x <- ck_create_testdata()

# create some 0/1 variables that should be perturbed later
x[, cnt_females := ifelse(sex == "male", 0, 1)]
x[, cnt_males := ifelse(sex == "male", 1, 0)]
x[, cnt_highincome := ifelse(income >= 9000, 1, 0)]
# a variable with positive and negative contributions
x[, mixed := sample(-10:10, nrow(x), replace = TRUE)]

# create record keys
x$rkey <- ck_generate_rkeys(dat = x)

# define required inputs

# hierarchy with some bogus codes
d_sex <- hier_create(root = "Total", nodes = c("male", "female"))
d_sex <- hier_add(d_sex, root = "female", "f")
d_sex <- hier_add(d_sex, root = "male", "m")

d_age <- hier_create(root = "Total", nodes = paste0("age_group", 1:6))
d_age <- hier_add(d_age, root = "age_group1", "ag1a")
d_age <- hier_add(d_age, root = "age_group2", "ag2a")

# define the cell key object
countvars <- c("cnt_females", "cnt_males", "cnt_highincome")
numvars <- c("expend", "income", "savings", "mixed")
tab <- ck_setup(
  x = x,
  rkey = "rkey",
  dims = list(sex = d_sex, age = d_age),
  w = "sampling_weight",
  countvars = countvars,
  numvars = numvars)

# show some information about this table instance
tab$print() # identical with print(tab)

# information about the hierarchies
tab$hierarchy_info()

# which variables have been defined?
tab$allvars()

# count variables
tab$cntvars()

# continuous variables
tab$numvars()

# create perturbation parameters for "total" variable and
# write to yaml-file

# create a ptable using functionality from the ptable-pkg
f_yaml <- tempfile(fileext = ".yaml")
p_cnts1 <- ck_params_cnts(
  ptab = ptable::pt_ex_cnts(),
  path = f_yaml)

# read parameters from yaml-file and set them for variable `"total"`
p_cnts1 <- ck_read_yaml(path = f_yaml)

tab$params_cnts_set(val = p_cnts1, v = "total")

# create alternative perturbation parameters by specifying parameters
para2 <- ptable::create_cnt_ptable(
  D = 8, V = 3, js = 2, create = FALSE)

p_cnts2 <- ck_params_cnts(ptab = para2)

# use these ptable it for the remaining variables
tab$params_cnts_set(val = p_cnts2, v = countvars)

# perturb a variable
tab$perturb(v = "total")

# multiple variables can be perturbed as well
tab$perturb(v = c("cnt_males", "cnt_highincome"))

# return weighted and unweighted results
tab$freqtab(v = c("total", "cnt_males"))

# numerical variables (positive variables using flex-function)
# we also write the config to a yaml file
f_yaml <- tempfile(fileext = ".yaml")

# create a ptable using functionality from the ptable-pkg
# a single ptable for all cells
ptab1 <- ptable::pt_ex_nums(parity = TRUE, separation = FALSE)

# a single ptab for all cells except for very small ones
ptab2 <- ptable::pt_ex_nums(parity = TRUE, separation = TRUE)

# different ptables for cells with even/odd number of contributors
# and very small cells
ptab3 <- ptable::pt_ex_nums(parity = FALSE, separation = TRUE)

p_nums1 <- ck_params_nums(
  ptab = ptab1,
  type = "top_contr",
  top_k = 3,
  mult_params = ck_flexparams(
    fp = 1000,
    p = c(0.30, 0.03),
    epsilon = c(1, 0.5, 0.2),
    q = 3),
  mu_c = 2,
  same_key = FALSE,
  use_zero_rkeys = FALSE,
  path = f_yaml)

# we read the parameters from the yaml-file
p_nums1 <- ck_read_yaml(path = f_yaml)

# for variables with positive and negative values
p_nums2 <- ck_params_nums(
  ptab = ptab2,
  type = "top_contr",
  top_k = 3,
  mult_params = ck_flexparams(
    fp = 1000,
    p = c(0.15, 0.02),
    epsilon = c(1, 0.4, 0.15),
    q = 3),
  mu_c = 2,
  same_key = FALSE)

# simple perturbation parameters (not using the flex-function approach)
p_nums3 <- ck_params_nums(
  ptab = ptab3,
  type = "mean",
  mult_params = ck_simpleparams(p = 0.25),
  mu_c = 2,
  same_key = FALSE)

# use `p_nums1` for all variables
tab$params_nums_set(p_nums1, c("savings", "income", "expend"))

# use different parameters for variable `mixed`
tab$params_nums_set(p_nums2, v = "mixed")

# identify sensitive cells to which extra protection (`mu_c`) is added.
tab$supp_p(v = "income", p = 85)
tab$supp_pq(v = "income", p = 85, q = 90)
tab$supp_nk(v = "income", n = 2, k = 90)
tab$supp_freq(v = "income", n = 14, weighted = FALSE)
tab$supp_val(v = "income", n = 10000, weighted = TRUE)
tab$supp_cells(
  v = "income",
  inp = data.frame(
    sex = c("female", "female"),
    "age" = c("age_group1", "age_group3")
  )
)

# perturb variables
tab$perturb(v = c("income", "savings"))

# extract results
tab$numtab("income", mean_before_sum = TRUE)
tab$numtab("income", mean_before_sum = FALSE)
tab$numtab("savings")

# results can be resetted, too
tab$reset_cntvars(v = "cnt_males")

# we can then set other parameters and perturb again
tab$params_cnts_set(val = p_cnts1, v = "cnt_males")

tab$perturb(v = "cnt_males")

# write results to a .csv file
tab$freqtab(
  v = c("total", "cnt_males"),
  path = file.path(tempdir(), "outtab.csv")
)

# show results containing weighted and unweighted results
tab$freqtab(v = c("total", "cnt_males"))

# utility measures for a count variable
tab$measures_cnts(v = "total", exclude_zeros = TRUE)

# modifications for perturbed count variables
tab$mod_cnts()

# display a summary about utility measures
tab$summary()
}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-cellkey_obj-new}{\code{ck_class$new()}}
\item \href{#method-cellkey_obj-perturb}{\code{ck_class$perturb()}}
\item \href{#method-cellkey_obj-freqtab}{\code{ck_class$freqtab()}}
\item \href{#method-cellkey_obj-numtab}{\code{ck_class$numtab()}}
\item \href{#method-cellkey_obj-measures_cnts}{\code{ck_class$measures_cnts()}}
\item \href{#method-cellkey_obj-measures_nums}{\code{ck_class$measures_nums()}}
\item \href{#method-cellkey_obj-allvars}{\code{ck_class$allvars()}}
\item \href{#method-cellkey_obj-cntvars}{\code{ck_class$cntvars()}}
\item \href{#method-cellkey_obj-numvars}{\code{ck_class$numvars()}}
\item \href{#method-cellkey_obj-hierarchy_info}{\code{ck_class$hierarchy_info()}}
\item \href{#method-cellkey_obj-mod_cnts}{\code{ck_class$mod_cnts()}}
\item \href{#method-cellkey_obj-mod_nums}{\code{ck_class$mod_nums()}}
\item \href{#method-cellkey_obj-supp_freq}{\code{ck_class$supp_freq()}}
\item \href{#method-cellkey_obj-supp_val}{\code{ck_class$supp_val()}}
\item \href{#method-cellkey_obj-supp_cells}{\code{ck_class$supp_cells()}}
\item \href{#method-cellkey_obj-supp_p}{\code{ck_class$supp_p()}}
\item \href{#method-cellkey_obj-supp_pq}{\code{ck_class$supp_pq()}}
\item \href{#method-cellkey_obj-supp_nk}{\code{ck_class$supp_nk()}}
\item \href{#method-cellkey_obj-params_cnts_get}{\code{ck_class$params_cnts_get()}}
\item \href{#method-cellkey_obj-params_cnts_set}{\code{ck_class$params_cnts_set()}}
\item \href{#method-cellkey_obj-reset_cntvars}{\code{ck_class$reset_cntvars()}}
\item \href{#method-cellkey_obj-reset_numvars}{\code{ck_class$reset_numvars()}}
\item \href{#method-cellkey_obj-reset_allvars}{\code{ck_class$reset_allvars()}}
\item \href{#method-cellkey_obj-params_nums_get}{\code{ck_class$params_nums_get()}}
\item \href{#method-cellkey_obj-params_nums_set}{\code{ck_class$params_nums_set()}}
\item \href{#method-cellkey_obj-summary}{\code{ck_class$summary()}}
\item \href{#method-cellkey_obj-print}{\code{ck_class$print()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-new"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-new}{}}}
\subsection{Method \code{new()}}{
Create a new table instance
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$new(x, rkey, dims, w = NULL, countvars = NULL, numvars = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{an object coercible to a \code{data.frame}}

\item{\code{rkey}}{either a column name within \code{x} referring to a variable containing record keys
or a single integer(ish) number > \code{5} that referns to the number of digits for record keys that
will be generated internally.}

\item{\code{dims}}{a list containing slots for each variable that should be
tabulated. Each slot consists should be created/modified using \code{\link[sdcHierarchies:hier_create]{sdcHierarchies::hier_create()}},
\code{\link[sdcHierarchies:hier_add]{sdcHierarchies::hier_add()}} and other functionality from package \code{sdcHierarchies}.}

\item{\code{w}}{(character) a scalar character referring to a variable in \code{x} holding sampling
weights. If \code{w} is \code{NULL} (the default), all weights are assumed to be \code{1}}

\item{\code{countvars}}{(character) an optional vector containing names of binary (0/1 coded)
variables withing \code{x} that should be included in the problem instance.
These variables can later be perturbed.}

\item{\code{numvars}}{(character) an optional vector of numerical variables that can later be tabulated.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new \code{cellkey_obj} object. Such objects (internally) contain the fully computed
statistical tables given input microdata (\code{x}), the hierarchical definitionals (\code{dims}) as
well as the remaining inputs. Intermediate results are stored internally and can only be
modified / accessed via the exported public methods described below.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-perturb"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-perturb}{}}}
\subsection{Method \code{perturb()}}{
Perturb a count- or magnitude variable
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$perturb(v)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{name(s) of count- or magnitude variables that should be perturbed.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. Updated data can be accessed using other exported
methods like \verb{$freqtab()} or \verb{$numtab()}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-freqtab"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-freqtab}{}}}
\subsection{Method \code{freqtab()}}{
Extract results from already perturbed count variables as a
\code{data.table}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$freqtab(v = NULL, path = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a vector of variable names for count variables. If \code{NULL}
(the default), the results are returned for all available count
variables. For variables that have not yet perturbed, columns
\code{puwc} and \code{pwc} are filled with \code{NA}.}

\item{\code{path}}{if not \code{NULL}, a scalar character defining a (relative
or absolute) path to which the result table should be written. A \code{csv}
file will be generated and, if specified, \code{path} must have
".csv" as file-ending}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
This method returns a \code{data.table} containing all combinations of the dimensional variables in
the first n columns. Additionally, the following columns are shown:
\itemize{
\item \code{vname}: name of the perturbed variable
\item \code{uwc}: unweighted counts
\item \code{wc}: weighted counts
\item \code{puwc}: perturbed unweighted counts or \code{NA} if \code{vname} was not yet perturbed
\item \code{pwc}: perturbed weighted counts or \code{NA} if \code{vname} was not yet perturbed
}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-numtab"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-numtab}{}}}
\subsection{Method \code{numtab()}}{
Extract results from already perturbed continuous variables
as a \code{data.table}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$numtab(v = NULL, mean_before_sum = FALSE, path = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a vector of variable names of continuous variables. If \code{NULL}
(the default), the results are returned for all available numeric variables.}

\item{\code{mean_before_sum}}{(logical); if \code{TRUE}, the perturbed values are adjusted
by a factor \verb{((n+p))⁄n} with
\itemize{
\item \code{n}: the original weighted cell value
\item \code{p}: the perturbed cell value
}

This makes sense if the the accuracy of the variable mean is considered to be
more important than accuracy of sums of the variable. The default value is
\code{FALSE} (no adjustment is done)}

\item{\code{path}}{if not \code{NULL}, a scalar character defining a (relative or absolute)
path to which the result table should be written. A \code{csv} file will be generated
and, if specified, \code{path} must have ".csv" as file-ending}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
This method returns a \code{data.table} containing all combinations of the
dimensional variables in the first n columns. Additionally, the following
columns are shown:
\itemize{
\item \code{vname}: name of the perturbed variable
\item \code{uws}: unweighted sum of the given variable
\item \code{ws}: weighted cellsum
\item \code{pws}: perturbed weighted sum of the given cell or \code{NA} if \code{vname}
has not not perturbed
}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-measures_cnts"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-measures_cnts}{}}}
\subsection{Method \code{measures_cnts()}}{
Utility measures for perturbed count variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$measures_cnts(v, exclude_zeros = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{name of a count variable for which utility measures
should be computed.}

\item{\code{exclude_zeros}}{should empty (zero) cells in the original values
be excluded when computing distance measures}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
This method returns a \code{list} containing a set of utility
measures based on some distance functions. For a detailed description
of the computed measures, see \code{\link[=ck_cnt_measures]{ck_cnt_measures()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-measures_nums"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-measures_nums}{}}}
\subsection{Method \code{measures_nums()}}{
Utility measures for continuous variables (not yet implemented)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$measures_nums(v)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{name of a continuous variable for which utility measures
should be computed.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
for (now) an empty list; In future versions of the package, the
Method will return utility measures for perturbed magnitude tables.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-allvars"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-allvars}{}}}
\subsection{Method \code{allvars()}}{
Names of variables that can be perturbed / tabulated
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$allvars()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
returns a \code{list} with the following two elements:
\itemize{
\item \code{cntvars}: character vector with names of available count
variables for perturbation
\item \code{numvars}: character vector with names of available numerical
variables for perturbation
}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-cntvars"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-cntvars}{}}}
\subsection{Method \code{cntvars()}}{
Names of count variables that can be perturbed
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$cntvars()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a character vector containing variable names
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-numvars"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-numvars}{}}}
\subsection{Method \code{numvars()}}{
Names of continuous variables that can be perturbed
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$numvars()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a character vector containing variable names
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-hierarchy_info"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-hierarchy_info}{}}}
\subsection{Method \code{hierarchy_info()}}{
Information about hierarchies
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$hierarchy_info()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a \code{list} (for each dimensional variable) with
information on the hierarchies. This may be used to restrict output tables to
specific levels or codes. Each list element is a \code{data.table} containing
the following variables:
\itemize{
\item \code{code}: the name of a code within the hierarchy
\item \code{level}: number defining the level of the code; the higher the number,
the lower the hierarchy with \code{1} being the overall total
\item \code{is_leaf}: if \code{TRUE}, this code is a leaf node which means no other codes
contribute to it
\item \code{parent}: name of the parent code
}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-mod_cnts"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-mod_cnts}{}}}
\subsection{Method \code{mod_cnts()}}{
Modifications applied to count variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$mod_cnts()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a \code{data.table} containing modifications applied to count variables
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-mod_nums"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-mod_nums}{}}}
\subsection{Method \code{mod_nums()}}{
Modifications applied to numerical variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$mod_nums()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a \code{data.table} containing modifications applied to numerical variables
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-supp_freq"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-supp_freq}{}}}
\subsection{Method \code{supp_freq()}}{
Identify sensitive cells based on minimum frequency rule
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$supp_freq(v, n, weighted = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a single variable name of a continuous variable (see method \code{numvars()})}

\item{\code{n}}{a number defining the threshold. All cells \verb{<= n} are considered as unsafe.}

\item{\code{weighted}}{if \code{TRUE}, the weighted number of contributors to a cell are compared to
the threshold specified in \code{n} (default); else the unweighted number of contributors is used.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-supp_val"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-supp_val}{}}}
\subsection{Method \code{supp_val()}}{
Identify sensitive cells based on weighted or unweighted cell value
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$supp_val(v, n, weighted = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a single variable name of a continuous variable (see method \code{numvars()})}

\item{\code{n}}{a number defining the threshold. All cells \verb{<= n} are considered as unsafe.}

\item{\code{weighted}}{if \code{TRUE}, the weighted cell value of variable \code{v} is compared to
the threshold specified in \code{n} (default); else the unweighted number is used.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-supp_cells"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-supp_cells}{}}}
\subsection{Method \code{supp_cells()}}{
Identify sensitive cells based on their \code{names}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$supp_cells(v, inp)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a single variable name of a continuous variable (see method \code{numvars()})}

\item{\code{inp}}{a \code{data.frame} where each colum represents a dimensional variable. Each row of
this input is then used to compute the relevant cells to be identified as sensitive where
\code{NA}-values are possible and used to match any characteristics of the dimensional variable.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-supp_p"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-supp_p}{}}}
\subsection{Method \code{supp_p()}}{
Identify sensitive cells based on the p\%-rule rule. Please note that this rule
can only be applied to positive-only variables.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$supp_p(v, p)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a single variable name of a continuous variable (see method \code{numvars()})}

\item{\code{p}}{a number defining a percentage between \code{1} and \code{99}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-supp_pq"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-supp_pq}{}}}
\subsection{Method \code{supp_pq()}}{
Identify sensitive cells based on the pq-rule. Please note that this rule
can only be applied to positive-only variables.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$supp_pq(v, p, q)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a single variable name of a continuous variable (see method \code{numvars()})}

\item{\code{p}}{a number defining a percentage between \code{1} and \code{99}.}

\item{\code{q}}{a number defining a percentage between \code{1} and \code{99}. This value must be larger than \code{p}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-supp_nk"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-supp_nk}{}}}
\subsection{Method \code{supp_nk()}}{
Identify sensitive cells based on the nk-dominance rule. Please note that this rule
can only be applied to positive-only variables.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$supp_nk(v, n, k)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a single variable name of a continuous variable (see method \code{numvars()})}

\item{\code{n}}{an integerish number \verb{>= 2}}

\item{\code{k}}{a number defining a percentage between \code{1} and \code{99}. All cells to which the top \code{n}
contributers contribute more than \verb{k\%} is considered unsafe}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-params_cnts_get"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-params_cnts_get}{}}}
\subsection{Method \code{params_cnts_get()}}{
Return perturbation parameters of count variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$params_cnts_get()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a named list in which each list-element contains the
active perturbation parameters for the specific count variable
defined by the list-name.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-params_cnts_set"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-params_cnts_set}{}}}
\subsection{Method \code{params_cnts_set()}}{
Set perturbation parameters for count variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$params_cnts_set(val, v = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{val}}{a perturbation object created with \code{\link[=ck_params_cnts]{ck_params_cnts()}}}

\item{\code{v}}{a character vector (or \code{NULL}). If \code{NULL} (the default),
the perturbation parameters provided in \code{val} are set for all
count variables; otherwise one may specify the names of
the count variables for which the parameters should be set.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other
methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-reset_cntvars"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-reset_cntvars}{}}}
\subsection{Method \code{reset_cntvars()}}{
reset results and parameters for already perturbed
count variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$reset_cntvars(v = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{if \code{v} equals \code{NULL} (the default), the results are reset
for all perturbed count variables; otherwise it is possible to specify
the names of already perturbed count variables.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other
methods (e.g \verb{$perturb()} or \verb{$freqtab()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-reset_numvars"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-reset_numvars}{}}}
\subsection{Method \code{reset_numvars()}}{
reset results and parameters for already perturbed
numerical variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$reset_numvars(v = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{if \code{v} equals \code{NULL} (the default), the results are reset for all perturbed
numerical variables; otherwise it is possible to specify the names of already
perturbed continuous variables.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other
methods (e.g \verb{$perturb()} or \verb{$numtab()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-reset_allvars"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-reset_allvars}{}}}
\subsection{Method \code{reset_allvars()}}{
reset results and parameters for all already perturbed variables.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$reset_allvars()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other
methods (e.g \verb{$perturb()}, \verb{$freqtab()} or \verb{$numtab()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-params_nums_get"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-params_nums_get}{}}}
\subsection{Method \code{params_nums_get()}}{
Return perturbation parameters of continuous variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$params_nums_get()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a named list in which each list-element contains the
active perturbation parameters for the specific continuous variable
defined by the list-name.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-params_nums_set"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-params_nums_set}{}}}
\subsection{Method \code{params_nums_set()}}{
set perturbation parameters for continuous variables.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$params_nums_set(val, v = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{val}}{a perturbation object created with \code{\link[=ck_params_nums]{ck_params_nums()}}}

\item{\code{v}}{a character vector (or \code{NULL}); if \code{NULL} (the default), the
perturbation parameters provided in \code{val} are set for all continuous
variables; otherwise one may specify the names of the numeric variables for
which the parameters should be set.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A modified \code{cellkey_obj} object in which private slots were
updated for side-effects. These updated values are used by other
methods (e.g \verb{$perturb()}).
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-summary"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-summary}{}}}
\subsection{Method \code{summary()}}{
some aggregated summary statistics about perturbed variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$summary()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
invisible \code{NULL}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cellkey_obj-print"></a>}}
\if{latex}{\out{\hypertarget{method-cellkey_obj-print}{}}}
\subsection{Method \code{print()}}{
prints information about the current table
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{ck_class$print()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
invisible \code{NULL}
}
}
}
