% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TextualCIU.R
\name{ciu.textual}
\alias{ciu.textual}
\title{Give textual CIU explanation}
\usage{
ciu.textual(
  ciu,
  instance = NULL,
  ind.inputs = NULL,
  ind.output = 1,
  in.min.max.limits = NULL,
  n.samples = 100,
  neutral.CU = 0.5,
  show.input.values = TRUE,
  concepts.to.explain = NULL,
  target.concept = NULL,
  target.ciu = NULL,
  ciu.meta = NULL,
  sort = "CI",
  n.features = NULL,
  use.text.effects = FALSE,
  CI.voc = data.frame(limits = c(0.2, 0.4, 0.6, 0.8, 1), texts = c("not important",
    "slightly important", "important", "very important", "extremely important")),
  CU.voc = data.frame(limits = c(0.2, 0.4, 0.6, 0.8, 1), texts = c("very bad", "bad",
    "average", "good", "very good"))
)
}
\arguments{
\item{ciu}{\code{ciu} object as created with \link{ciu} function (not to be confused
with \code{CIU} object as created by \link{ciu.new}).}

\item{instance}{Input values for the instance to explain. Should be a
\link{data.frame} even though a \code{vector} or \code{matrix} might work too if input
names and other needed metadata can be deduced from the dataset or other
parameters given to \code{\link{ciu.new}}.}

\item{ind.inputs}{Indices of input features to explain (the set \{i\} in CIU
formulae)}

\item{ind.output}{Index of output to be explained.}

\item{in.min.max.limits}{\link{data.frame} or \link{matrix} with one row per output
and two columns, where the first column indicates the minimal value and the
second column the maximal value for that output. ONLY NEEDED HERE IF not
given as parameter to \link{ciu.new} or if the limits are different for this
specific instance than the default ones.}

\item{n.samples}{How many instances to generate for estimating CI and CU.
For inputs of type \link{factor}, all possible combinations of input values
are generated, so this parameter only influences how many instances are
(at least) generated for continuous-valued inputs.}

\item{neutral.CU}{Indicates when the Contextual Utility is considered
to be "negative". The default value of 0.5 seems quite logical for most cases.}

\item{show.input.values}{Include input values after input labels or not.
Default is TRUE.}

\item{concepts.to.explain}{List of input feature concepts to explain, as defined
by vocabulary provided as argument to \link{ciu.new}. If \code{ind.inputs=NULL},
then use \code{concepts.to.explain} instead. If both are \code{NULL}, then use all inputs.}

\item{target.concept}{If provided, then calculate CIU of inputs
\code{ind.inputs.to.explain} relative to the given concept rather than
relative to the actual output(s). \code{ind.inputs.to.explain} should
normally be a subset (or all) of the inputs that \code{target.concept}
consists of, even though that not required by the CIU calculation.
If a \code{target.ciu} is provided, then the \code{target.concept} doesn't have to
be included in the \code{vocabulary} gives as parameter to \link{ciu.new}
(at least for the moment).}

\item{target.ciu}{\code{ciu.result} object previously calculated for
\code{target.concept}. If a \code{target.concept} is provided but \code{target.ciu=NULL},
then \code{target.ciu} is estimated by a call to \link{ciu.explain} with the
\code{n.samples} value given as a parameter to this call. It may be useful
to provide \code{target.ciu} if it should be estimated using some other
(typically greater) value for \code{n.samples} than the default one, or if it
has already been calculated for some reason.}

\item{ciu.meta}{If given, then use existing \code{ciu.meta.result} rather
than calling \link{ciu.meta.explain}.}

\item{sort}{NULL, "CI" or "CU".}

\item{n.features}{Maximal number of features to include in explanation.}

\item{use.text.effects}{Add bold/italics/colors effects?}

\item{CI.voc}{Limits and texts to use for CI values.}

\item{CU.voc}{Limits and texts to use for CU values.}
}
\value{
Text string with explanation.
}
\description{
Provide textual CIU explanations as those used in Kary Främling's
PhD thesis.
}
\examples{
# Explaining the classification of an Iris instance with lda model.
# We use a versicolor (instance 100).
library(MASS)
test.ind <- 100
iris_test <- iris[test.ind, 1:4]
iris_train <- iris[-test.ind, 1:4]
iris_lab <- iris[[5]][-test.ind]
model <- lda(iris_train, iris_lab)

# Create CIU object
ciu <- ciu.new(model, Species~., iris)

# Give textual explanation. Use 'cat' for getting newlines to work.
cat(ciu.textual(ciu, iris_test, ind.output = 2))
cat(ciu.textual(ciu, iris_test, ind.output = 2, n.features = 2))

\dontrun{
# Boston housing, GBM model.
library(caret)
kfoldcv <- trainControl(method="cv", number=10)
gbm <- train(medv ~ ., Boston, method="gbm", trControl=kfoldcv)
boston.inst <- Boston[370,1:13]
ciu <- ciu.new(gbm, medv~., Boston)
cat(ciu.textual(ciu, boston.inst,use.text.effects = TRUE))
# Customized limits for CI.
cat(ciu.textual(ciu, boston.inst,use.text.effects = TRUE,
  CI.voc = data.frame(limits=c(0.05,0.1,0.3,0.5,1.0),
texts=c("not important","little important", "important","very important",
  "extremely important"))))

# Intermediate concepts
social<-c(1,11,13); usage_type<-c(2,3); chas<-c(4); air_quality<-c(5)
housing<-c(6,7); transport<-c(8,9); blacks<-c(12); tax<-c(10)
Boston.voc <- list("SOCIAL"=social, "LAND USAGE"=usage_type, "Charles R. dummy"=chas,
"Air quality (Nox)"=air_quality, "HOUSING"=housing, "TRANSPORT"=transport,
"Prop. of black people"=blacks, "Tax"=tax)
ciu <- ciu.new(gbm, medv~., Boston, vocabulary = Boston.voc)

# We use `meta.explain` here to avoid differences due to sampling.
meta.top <- ciu$meta.explain(boston.inst, concepts.to.explain=names(Boston.voc))
cat(ciu.textual(ciu, boston.inst, use.text.effects = TRUE, ciu.meta = meta.top))

# Explain intermediate concept utility, using input features (could also
# be using other intermediate concepts).
cat(ciu.textual(ciu, boston.inst, use.text.effects = TRUE, ind.inputs = Boston.voc$SOCIAL,
  target.concept = "SOCIAL", target.ciu = meta.top$ciuvals[["SOCIAL"]]))
cat(ciu.textual(ciu, boston.inst, use.text.effects = TRUE, ind.inputs = Boston.voc$HOUSING,
  target.concept = "HOUSING", target.ciu = meta.top$ciuvals[["HOUSING"]]))
}
}
