##' Variable Visualizations for Descriptive Tables
##'
##' Using as underlying functions the plot functions from
##' \code{ggstatsplot} this package visualizes the variables in a
##' descriptive table generated via
##' \code{\link[descsuppR]{buildDescrTbl}}.
##'
##' @param dtab \code{data.frame} with attributes
##'   \code{descrtbl_full}, \code{data}, and \code{groupby} as
##'   generated by \code{\link[descsuppR]{buildDescrTbl}}
##' @param plotfuns named list of tibbles.  names must be in \code{dtab$parameter}.  entries must be of the form \code{tibble(plotfun = "betweenstats", args = list(list(messages = FALSE, type = "nonparametric")))}.  For missing entries \code{plotDescTbl} tries to guess a suitable plot function.  Defaults to \code{NULL} meaning that the plot functions for all parameters are guessed.
##' @param verbose logical or integer.  Level of verbosity.  If
##'   \code{TRUE} or > 0, print some messages.  Defaults to 0.
##' @param pairwise.display Decides if and which pairwise comparisons are shown in the plots.
##'   Available options are:
##'   - `"none"`
##'   - `"significant"` (abbreviation accepted: `"s"`)
##'   - `"non-significant"` (abbreviation accepted: `"ns"`)
##'   - `"all"`
##'   Defaults to "none".
##' @param pval.display logical.  If \code{TRUE} p-values are shown in the plots.
##' If \code{dtab} contains adjusted p-values, these are shown in the plots. Defaults to
##'   \code{FALSE}.
##' @param ... additional parameters will be passed to the plotting functions.
##' @return \code{tibble} with column \code{p} containing the plots.
##' @author Dr. Andreas Leha
##' @export
##' @examples
##' ttt <- data.frame(data="training set",
##'                   age=runif(100, 0, 100),
##'                   sex=as.factor(sample(c("m","f"), 100, replace=TRUE, prob=c(0.3, 0.7))),
##'                   score=factor(sample(1:5, 100, replace=TRUE),
##'                     ordered=TRUE,
##'                     levels=1:5))
##' ttt2 <- data.frame(data="test set",
##'                    age=runif(100, 0, 100),
##'                    sex=as.factor(sample(c("m","f"), 100, replace=TRUE, prob=c(0.5,0.5))),
##'                    score=factor(sample(1:5, 100, replace=TRUE),
##'                      ordered=TRUE,
##'                      levels=1:5))
##'
##' ## ungrouped table and plots
##' dtab <- buildDescrTbl(dplyr::mutate(rbind(ttt, ttt2), data = as.factor(data)),
##'                       includeNAs=TRUE,
##'                       dopvals = TRUE)
##'
##' plotDescTbl(dtab)
##'
##' ## grouped table and plots
##' dtab <- buildDescrTbl(dplyr::mutate(rbind(ttt, ttt2), data = as.factor(data)),
##'                       groupby="data",
##'                       includeNAs=TRUE,
##'                       dopvals = TRUE)
##'
##' plotDescTbl(dtab)
plotDescTbl <- function(dtab, plotfuns = NULL, verbose = 0, pairwise.display = "none", pval.display = FALSE, ...)
{

  dat  <- attr(dtab, "data")
  groupby <- attr(dtab, "groupby")
  dtab <- attr(dtab, "descrtbl_full")

  ## ggstatsplot chokes on some variable names
  ## all reserved names?
  ## at least 'data' is not possible as name of the grouping variable
  colnames(dat) <- gsub("^data$", "thedata", colnames(dat))
  if (!is.null(groupby)) groupby <- gsub("^data$", "thedata", groupby)
  dtab <- dtab %>% dplyr::mutate(parameter = gsub("^data$", "thedata", .data$parameter))

  ## setup results data.frame 'tests' and ensure it has a column 'test'
  tests <-
    dtab %>%
    dplyr::filter(.data$parameter != "", !duplicated(.data$parameter), .data$parameter != "n") %>%
    dplyr::select(dplyr::one_of("parameter",
                                "p value",
                                "adjusted p value",
                                "test"))
  if ("test" %in% colnames(tests))
    tests <- tests %>% dplyr::filter(.data$test != "")
  ## pretty format p values
  if("p value" %in% colnames(tests))
    tests <- tests %>%
      dplyr::mutate(`p value` =  `p value` %>% as.numeric() %>% descutils::prettyPvalues(digits = 3, orgbold = FALSE, lhs = "p"))
  if("adjusted p value" %in% colnames(tests))
    tests <- tests %>%
      dplyr::mutate(`adjusted p value` =  `adjusted p value` %>% as.numeric() %>% descutils::prettyPvalues(digits = 3, orgbold = FALSE, lhs = "p"))
  ## test suggestions
  tests <-
    tests %>%
    dplyr::group_by(.data$parameter) %>%
    ## TODO use descsuppR::suggestTest()
    dplyr::mutate(test_suggestion = ifelse(is.numeric(dat[[.data$parameter[1]]]),
                                           "Welch Two Sample t-test",
                                           "Fisher's Exact Test for Count Data"))
  ## use test_suggestion only when there is no test given
  if (!"test" %in% colnames(tests))
    tests$test <- "<untested>"
  tests <-
    tests %>%
    dplyr::mutate(test = ifelse(.data$test == "<untested>", .data$test_suggestion, .data$test)) %>%
    dplyr::select(-.data$test_suggestion)

  ## choose plot functions based on 'test' column
  tests <-
    tests %>%
    dplyr::group_by_all() %>%
    dplyr::do({
             if (!is.null(plotfuns) && .$parameter %in% names(plotfuns)) {
               plotfuns[[.$parameter]]
             } else {
               suggestPlot(.$test)
             }})
  if (verbose) tests %>% print

  ## call the chosen plot functions to generate plots in the 'p' column
  tests <-
    tests %>%
      dplyr::do(p = {
        ldat <- .data
        if (verbose) ldat$parameter %>% print
        p <- do.call(ldat$plotfun, c(list(data = dat,
                                          parameter = ldat$parameter,
                                          groupby = groupby,
                                          pval.display = pval.display,
                                          ldat = ldat),
                                     unlist(ldat$args, recursive = FALSE),
                                     list(...)))

        p
      })

  ## fix the order
  params_in_order <- dtab %>% dplyr::filter(parameter != "n") %>% .$parameter %>% unique
  tests <-
    tests %>%
    dplyr::mutate(parameter = factor(parameter, levels = params_in_order)) %>%
    dplyr::mutate(paramnumber = as.numeric(parameter)) %>%
    dplyr::arrange(paramnumber) %>%
    dplyr::select(-paramnumber)

  return(tests)
}
