% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/difORD.R
\name{difORD}
\alias{difORD}
\title{DIF detection among ordinal data.}
\usage{
difORD(Data, group, focal.name, model = "adjacent", type = "both", match = "zscore",
       anchor = NULL, purify = FALSE, nrIter = 10, p.adjust.method = "none",
       alpha = 0.05, parametrization)
}
\arguments{
\item{Data}{data.frame or matrix: dataset which rows represent
ordinally scored examinee answers and columns correspond to the
items. In addition, \code{Data} can hold the vector of group
membership.}

\item{group}{numeric or character: a dichotomous vector of the same
length as \code{nrow(Data)} or a column identifier of
\code{Data}.}

\item{focal.name}{numeric or character: indicates the level of
\code{group} which corresponds to focal group.}

\item{model}{character: logistic regression model for ordinal data
(either \code{"adjacent"} (default) or \code{"cumulative"}). See
\strong{Details}.}

\item{type}{character: type of DIF to be tested. Either
\code{"both"} for uniform and non-uniform DIF (default), or
\code{"udif"} for uniform DIF only, or \code{"nudif"} for
non-uniform DIF only. Can be specified as a single value (for all
items) or as an item-specific vector.}

\item{match}{character or numeric: matching criterion to be used as an
estimate of the trait. It can be either \code{"zscore"} (default;
standardized total score), \code{"score"} (total test score),
\code{"restscore"} (total score without the tested item),
\code{"zrestscore"} (standardized total score without the tested item), a
numeric vector of the same length as a number of observations in the
\code{Data}, or a numeric matrix of the same dimensions as \code{Data}
(each column represents matching criterion for one item).}

\item{anchor}{character or numeric: specification of DIF-free (anchor) items
used to compute the matching criterion (\code{match}). Can be either
\code{NULL} (default; all items are used for the calculation), or a vector
of item identifiers (integers indicating column numbers or item names in
`Data`) specifying which items are currently considered as anchor items.
This argument is ignored if the \code{match} is not \code{"zscore"},
\code{"score"}, \code{"restscore"}, or \code{"zrestscore"}. For \code{match =
"score"} or \code{match = "zscore"}, the matching criterion is computed
from the items specified in the anchor set. For \code{match = "restscore"} or
\code{match = "zrestscore"}, the same anchor items are used, except that the
item currently under test is excluded from the computation.}

\item{purify}{logical: should the item purification be applied? (default is
\code{FALSE}). Item purification is not applied when set of anchor items in
\code{anchor} is specified or when \code{match} is not \code{"zscore"},
\code{"score"}, \code{"restscore"}, or \code{"zrestscore"}.}

\item{nrIter}{numeric: the maximal number of iterations in the item
purification (default is 10).}

\item{p.adjust.method}{character: method for multiple comparison
correction. Possible values are \code{"holm"}, \code{"hochberg"},
\code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"},
\code{"fdr"}, and \code{"none"} (default). For more details see
\code{\link[stats]{p.adjust}}.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{parametrization}{deprecated. Use
\code{\link[difNLR]{coef.difORD}} for different
parameterizations.}
}
\value{
The \code{difORD()} function returns an object of class
  \code{"difORD"}. The output including values of the test
  statistics, p-values, and items marked as DIF is displayed by the
  \code{print()} method.

A list of class \code{"difORD"} with the following arguments:
\describe{
  \item{\code{Sval}}{the values of likelihood ratio test statistics.}
  \item{\code{ordPAR}}{the estimates of the final model.}
  \item{\code{ordSE}}{standard errors of the estimates of the final model.}
  \item{\code{parM0}}{the estimates of null model.}
  \item{\code{parM1}}{the estimates of alternative model.}
  \item{\code{llM0}}{log-likelihood of null model.}
  \item{\code{llM1}}{log-likelihood of alternative model.}
  \item{\code{AICM0}}{AIC of null model.}
  \item{\code{AICM1}}{AIC of alternative model.}
  \item{\code{BICM0}}{BIC of null model.}
  \item{\code{BICM1}}{BIC of alternative model.}
  \item{\code{DIFitems}}{either the column identifiers of the items which were detected as DIF, or
  \code{"No DIF item detected"} in case no item was detected as DIF.}
  \item{\code{model}}{model used for DIF detection.}
  \item{\code{type}}{character: type of DIF that was tested.}
  \item{\code{anchor}}{DIF free items specified by the \code{anchor} and \code{purify}.}
  \item{\code{purification}}{\code{purify} value.}
  \item{\code{nrPur}}{number of iterations in item purification process. Returned only if \code{purify}
  is \code{TRUE}.}
  \item{\code{difPur}}{a binary matrix with one row per iteration of item purification and one column per item.
  \code{"1"} in i-th row and j-th column means that j-th item was identified as DIF in i-th iteration. Returned only
  if \code{purify} is \code{TRUE}.}
  \item{\code{conv.puri}}{logical indicating whether item purification process converged before the maximal number
  \code{nrIter} of iterations. Returned only if \code{purify} is \code{TRUE}.}
  \item{\code{p.adjust.method}}{character: method for multiple comparison correction which was applied.}
  \item{\code{pval}}{the p-values by likelihood ratio test.}
  \item{\code{adj.pval}}{the adjusted p-values by likelihood ratio test using \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of likelihood ratio test.}
  \item{\code{alpha}}{numeric: significance level.}
  \item{\code{Data}}{the data matrix.}
  \item{\code{group}}{the vector of group membership.}
  \item{\code{group.names}}{levels of grouping variable.}
  \item{\code{match}}{matching criterion.}
  \item{\code{match.name}}{Name of the matching criterion.}
  }

For an object of class \code{"difORD"} several methods are available (e.g., \code{methods(class = "difORD")}).
}
\description{
Performs DIF detection procedure for ordinal data
  based either on adjacent category logit model or on cumulative
  logit model and likelihood ratio test of a submodel.
}
\details{
Calculates DIF likelihood ratio statistics based either on adjacent
category logit model or on cumulative logit model for ordinal data.

Using adjacent category logit model, logarithm of ratio of
probabilities of two adjacent categories is
\deqn{log(P(y = k) / P(y = k - 1)) = b_0k + b_1 * x + b_2k * g + b_3 * x:g,}
where \eqn{x} is by default standardized total score (also called
Z-score) and \eqn{g} is a group membership.

Using cumulative logit model, probability of gaining at least
\eqn{k} points is given by 2PL model, i.e.,
\deqn{P(y >= k) = exp(b_0k + b_1 * x + b_2k * g + b_3 * x:g) / (1 + exp(b_0k + b_1 * x + b_2k * g + b_3 * x:g)).}
The category probability (i.e., probability of gaining exactly
\eqn{k} points) is then \eqn{P(y = k) = P(y >= k) - P(y >= k + 1)}.

Both models are estimated by iteratively reweighted least squares.
For more details see \code{\link[VGAM]{vglm}}.

Missing values are allowed but discarded for item estimation. They
must be coded as \code{NA} for both, \code{Data} and \code{group}
parameters.
}
\examples{
# loading data
data(Anxiety, package = "ShinyItemAnalysis")
Data <- Anxiety[, paste0("R", 1:29)] # items
group <- Anxiety[, "gender"] # group membership variable

# testing both DIF effects with adjacent category logit model
(x <- difORD(Data, group, focal.name = 1, model = "adjacent"))
\dontrun{
# graphical devices
plot(x, item = 6)
plot(x, item = "R6")
plot(x, item = "R6", group.names = c("Males", "Females"))

# estimated parameters
coef(x)
coef(x, SE = TRUE) # with SE
coef(x, SE = TRUE, simplify = TRUE) # with SE, simplified

# AIC, BIC, log-likelihood
AIC(x)
BIC(x)
logLik(x)

# AIC, BIC, log-likelihood for the first item
AIC(x, item = 1)
BIC(x, item = 1)
logLik(x, item = 1)

# testing both DIF effects with Benjamini-Hochberg adjustment method
difORD(Data, group, focal.name = 1, model = "adjacent", p.adjust.method = "BH")

# testing both DIF effects with item purification
difORD(Data, group, focal.name = 1, model = "adjacent", purify = TRUE)

# testing uniform DIF effects
difORD(Data, group, focal.name = 1, model = "adjacent", type = "udif")
# testing non-uniform DIF effects
difORD(Data, group, focal.name = 1, model = "adjacent", type = "nudif")

# testing both DIF effects with different matching criteria
ddfMLR(Data, group, focal.name = 1, model = "adjacent", match = "score")
difORD(Data, group, focal.name = 1, model = "adjacent", match = "restscore")
difORD(Data, group, focal.name = 1, key, match = "zrestscore")
match <- rowSums(GMAT[, 1:20])
difORD(Data, group, focal.name = 1, key, match = match)
match <- replicate(ncol(Data), GMAT$criterion)
difORD(Data, group, focal.name = 1, key, match = match)
match <- as.data.frame(match)
difORD(Data, group, focal.name = 1, key, match = match)

difORD(Data, group, focal.name = 1, model = "adjacent", match = "score")

# testing both DIF effects with cumulative logit model
(x <- difORD(Data, group, focal.name = 1, model = "cumulative"))
# graphical devices
plot(x, item = 7, plot.type = "cumulative")
plot(x, item = 7, plot.type = "category")

# estimated parameters
coef(x, simplify = TRUE)
}
}
\references{
Agresti, A. (2010). Analysis of ordinal categorical data. Second edition. John Wiley & Sons.

Hladka, A. (2021). Statistical models for detection of differential item functioning. Dissertation thesis.
Faculty of Mathematics and Physics, Charles University.

Hladka, A. & Martinkova, P. (2020). difNLR: Generalized logistic regression models for DIF and DDF detection.
The R Journal, 12(1), 300--323, \doi{10.32614/RJ-2020-014}.
}
\seealso{
\code{\link[difNLR]{plot.difORD}} for graphical representation of item characteristic curves. \cr
\code{\link[difNLR]{coef.difORD}} for extraction of item parameters with their standard errors. \cr
\code{\link[difNLR]{predict.difORD}} for calculation of predicted values. \cr
\code{\link[difNLR]{logLik.difORD}}, \code{\link[difNLR]{AIC.difORD}}, \code{\link[difNLR]{BIC.difORD}}
for extraction of log-likelihood and information criteria. \cr

\code{\link[stats]{p.adjust}} for multiple comparison corrections. \cr
\code{\link[VGAM]{vglm}} for estimation function using iteratively reweighted least squares.
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
\email{hladka@cs.cas.cz} \cr

Patricia Martinkova \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
\email{martinkova@cs.cas.cz} \cr
}
\keyword{DIF}
