# Function that tests if all inputs of eiopt2 are correct
tests_inputs_eiopt2 <- function(argg){

  argg.1 <- as.matrix(argg$votes.election1)
  argg.2 <- as.matrix(argg$votes.election2)

  # Test names 1
  clase <- class(argg$weights)
  if (clase %in% c("matrix", "numeric", "data.frame")){
    if(min(argg$weights) < 0) stop('Negative values are not allowed as "weights".')
    if(length(dim(argg$weights)) == 2 & nrow(argg$weights) != nrow(argg.1)){
      stop('The number of rows of the matrix "weights" must be the same as the number of rows in "votes.election1".')
    } else {
      if(length(argg$weights) != nrow(argg.1))
        stop('The length of the vector "weights" must be the same as the number of rows in "votes.election1".')
    }

  }
  if (clase == "character"){
    if (!(argg$weights[1L] %in% c("row", "constant", "size")))
      stop('The value set for argument "weights" is not allowed. The only allowed strings for "weights" are "row", "constant" and "size".')
  }

  # Test names 2
  if (!(argg$census.changes[1L] %in% c("adjust2", "raw", "simultaneous", "regular", "ordinary", "enriched",
                                       "adjust1", "semifull", "full", "fullreverse", "gold")))
    stop('The value set for argument "census.changes" is not allowed. The list of values avaliable can be consulted in the "Details" section of the help of the function.')

  # Tests numeric data
  x <- as.matrix(argg$votes.election1)
  y <- as.matrix(argg$votes.election2)
  if (ncol(x) == 1L) x <- t(x)
  if (ncol(y) == 1L) y <- t(y)
  if (nrow(x) != nrow(y))
    stop('The number of spatial units is different in "votes_election1" and "votes_election2".')
  if (argg$census.changes[1L] %in% c("semifull", "simultaneous", "full", "fullreverse", "gold")){
    if (!identical(round(rowSums(x)), round(rowSums(y)))){
      texto <- paste0('The number of voters (electors) in Election 1 and ',
                      'Election 2 differ (in at least a territorial unit). This is not ',
                      'allowed in a \"', argg$census.changes[1L], '\" scenario.')
      stop(texto)
    }
  }
  if (min(x,y) < 0L) stop('Negative values are not allowed in arguments "votes_election1" and "votes_election2".')

  # Test lambda
  if(argg$eps <= 0){
    stop('Only positive values are allowed for the "eps" argument.')
  if(argg$ilack.max < 1L)
      stop('Values smaller than 1 are not allowed for the "ilack.max" argument.')
  }
}
##------------------------------------------------

## Function to adjust x and y and, if proceed, compute net entries and exits
adjust_x_y <- function(votes.election1, votes.election2, census.changes){

  x <- as.matrix(votes.election1)
  y <- as.matrix(votes.election2)

  # Adjustment of x and y depending on the scenario
  if (census.changes == "adjust2"){
    x <- x*rowSums(y)/rowSums(x)
  }
  if (census.changes == "adjust1"){
    y <- y*rowSums(x)/rowSums(y)
  }
  # Estimation of net entries/exits in the election census
  if (census.changes %in% c("raw", "regular", "ordinary", "enriched")){
    NET_ENTRIES <- NET_EXITS <- rep(0L, nrow(x))
    d <- rowSums(y) - rowSums(x)
    if (any(d != 0L)) {
      NET_ENTRIES[d > 0L] <- d[d > 0L]
      NET_EXITS[d < 0L] <- -d[d < 0L]
    }
    # Net entries and exits
    if (sum(NET_ENTRIES) > 0L){
      x <- cbind(x, NET_ENTRIES)
    }
    if (sum(NET_EXITS) > 0L){
      y <- cbind(y, NET_EXITS)
    }
  }

  return(list("x" = x, "y" = y))
}

##------------------------------------------------

## function for defining the general global constraints of equality
global_equal_constraints <- function(x, y){ #x: origin; y: destination.
  nr <- ncol(x)
  nc <- ncol(y)

  # Equal constraints
  y.p <- colSums(y)#/sum(y)
  x.p <- colSums(x)#/sum(x)
  A1 <- kronecker(diag(1, nrow = nc), t(x.p)) # Matrix of column constraints
  A2 <- kronecker(t(rep(1, nc)), diag(nr)) # Matrix of unitary-row sum constraints
  A <- rbind(A1, A2)
  A <- A[-(nc + nr), ] # Redundant constraints
  B <- c(y.p, rep(1, nr - 1)) # Vector of coefficients of constraints

  return(list("A" = A, "B" = B))
}
##------------------------------------------------

# Function for defining the global constraints
global_constraints <- function(x, y, nr0, nc0, census.changes){
  nr <- ncol(x)
  nc <- ncol(y)

  # Initial global constraints
  gcont <- global_equal_constraints(x, y)

  # Additional global constraints
  if((census.changes %in% c("adjust1", "adjust2", "simultaneous", "raw")) |
     (census.changes == "regular" & nc0 == nc) |
     (census.changes == "ordinary" & nr0 == nr)
   )
  {
    A <- gcont$A
    B <- gcont$B
  }

  if((census.changes == "raw" & nr0 < nr & nc0 < nc) |
     (census.changes == "regular" & nr0 == nr & nc0 < nc) |
     (census.changes == "ordinary" & nr0 < nr & nc0 == nc) |
     (census.changes == "enriched" & nr0 == nr & nc0 == nc) |
     (census.changes == "semifull")
  )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A <- rbind(gcont$A, A3)
    B <- c(gcont$B, 0)
  }


  if((census.changes == "regular" & nr0 < nr & nc0 < nc) |
     (census.changes == "enriched" & nr0 < nr & nc0 == nc) |
     (census.changes == "full")
  )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A4 <- c(rep(0, nr * nc - 2), 1, 0) # Matrix for p_R-1,C = 0 constraint
    A <- rbind(gcont$A, A3, A4)
    B <- c(gcont$B, 0, 0)
  }

  if((census.changes == "ordinary" & nr0 < nr & nc0 < nc) |
     (census.changes == "enriched" & nr0 == nr & nc0 < nc) |
     (census.changes == "fullreverse")
  )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A4 <- c(rep(0, nr * (nc - 1) - 1), 1, rep(0, nr)) # Matrix for p_R,C-1 = 0 constraint
    A <- rbind(gcont$A, A3, A4)
    B <- c(gcont$B, 0, 0)
  }

  if((census.changes == "enriched" & nr0 < nr & nc0 < nc) |
     (census.changes == "gold")
    )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A4 <- c(rep(0, nr * nc - 2), 1, 0) # Matrix for p_R-1,C = 0 constraint
    A5 <- c(rep(0, nr * (nc - 1) - 1), 1, rep(0, nr)) # Matrix for p_R,C-1 = 0 constraint
    A6 <- c(rep(0, nr * (nc - 1) - 2), 1, 0, rep(0, nr)) # Matrix for p_R-1,C-1 = 0 constraint

    A <- rbind(gcont$A, A3, A4, A5, A6)
    B <- c(gcont$B, 0, 0, 0, 0)
  }

  # unuequal constraints
  A <- rbind(A, diag(nc*nr))
  B <- c(B, rep(0, nc*nr))

  return(list("A" = A, "B" = B))
}

##------------------------------------------------

# Function for defining the global objective function
global_obj <- function(x, y, weights){
  nr <- ncol(x)
  nc <- ncol(y)
  nu <- nrow(x)

  if(length(dim(weights)) == 2) weights <- "row"
  if(nrow(x) == length(weights)){
    x1 <- x/rowSums(x)
    y1 <- y/rowSums(y)
    Q <- matrix(0, nrow = nr, ncol = nr)
    d.lin <- rep(0, nc * nr)
    for (ii in 1L:nu){
      Q <- Q + weights[ii]*x1[ii, ]%*% t(x1[ii, ])
      d.lin <- d.lin + 2*weights[ii]*as.vector(x1[ii,]%*%t(y1[ii,]))
    }
    Q <- kronecker(diag(nc), 2*Q)
  } else {
    # row
      if(weights == "row"){
        Q <- kronecker(diag(nc), 2*t(x)%*%x)
        d.lin <- rep(0, nc*nr)
        for (ii in 1L:nu){
          d.lin <- d.lin + 2*as.vector(x[ii,]%*%t(y[ii,]))
        }
      }

      # constant
      if (weights == "constant"){
        x1 <- x/rowSums(x)
        y1 <- y/rowSums(y)
        Q <- kronecker(diag(nc), 2*t(x1)%*%x1)
        d.lin <- rep(0, nc*nr)
        for (ii in 1L:nu){
          d.lin <- d.lin + 2*as.vector(x1[ii,]%*%t(y1[ii,]))
        }
      }

      # size
      if (weights == "size"){
        weights <- rowSums(x)
        x1 <- x/rowSums(x)
        y1 <- y/rowSums(y)
        Q <- matrix(0, nrow = nr, ncol = nr)
        d.lin <- rep(0, nc*nr)
        for (ii in 1L:nu){
          Q <- Q + weights[ii]*x1[ii, ]%*% t(x1[ii, ])
          d.lin <- d.lin + 2*weights[ii]*as.vector(x1[ii,]%*%t(y1[ii,]))
        }
        Q <- kronecker(diag(nc), 2*Q)
      }
  }

  return(list("Q"= Q, "d.lin" = d.lin))

}
##------------------------------------------------

# Function for defining the matrix of weights

matrix_weights <- function(x, weights, nr0){

  m.weights <- x
  if (is.character(weights)){
    if(weights == "row") m.weights <- x
    if(weights == "constant") m.weights[,] <- 1
    if(weights == "size") m.weights <- kronecker(t(rep(1, ncol(x))), rowSums(x))
  }
  if(nrow(x) == length(weights)) m.weights <- kronecker(t(rep(1, ncol(x))), weights)

  if(length(dim(weights)) == 2L){
    m.weights <- as.matrix(weights)
     if(ncol(weights) != ncol(x)){
       r.x <- x[, (ncol(weights) + 1L):ncol(x)]
       r.x <- r.x/rowSums(x)
       m.weights <- cbind(as.matrix(weights), r.x*rowSums(weights))
     }
  }

  return(m.weights)
}
##------------------------------------------------

# Function for defining the unit equal constraints
unit_constraints <- function(xi, yi, nr0, nc0, census.changes){
  nr <- length(xi)
  nc <- length(yi)

  # Initial equal constraints
  # Matrix of first set of constraints
  A1 <- kronecker(diag(1, nrow = nc), t(xi))
  # Matrix of second set of constraints: unitary-row sums
  A2 <- kronecker(t(rep(1, nc)), diag(nr))
  A <- rbind(A1, A2)
  # Constraints
  B <- c(yi, rep(1, nr))

  # Additional global constraints
  if((census.changes == "raw" & nr0 < nr & nc0 < nc) |
     (census.changes == "regular" & nr0 == nr & nc0 < nc) |
     (census.changes == "ordinary" & nr0 < nr & nc0 == nc) |
     (census.changes == "enriched" & nr0 == nr & nc0 == nc) |
     (census.changes == "semifull")
  )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A <- rbind(A, A3)
    B <- c(B, 0)
  }

  if((census.changes == "regular" & nr0 < nr & nc0 < nc) |
     (census.changes == "enriched" & nr0 < nr & nc0 == nc) |
     (census.changes == "full")
  )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A4 <- c(rep(0, nr * nc - 2), 1, 0) # Matrix for p_R-1,C = 0 constraint
    A <- rbind(A, A3, A4)
    B <- c(B, 0, 0)
  }

  if((census.changes == "ordinary" & nr0 < nr & nc0 < nc) |
     (census.changes == "enriched" & nr0 == nr & nc0 < nc) |
     (census.changes == "fullreverse")
  )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A4 <- c(rep(0, nr * (nc - 1) - 1), 1, rep(0, nr)) # Matrix for p_R,C-1 = 0 constraint
    A <- rbind(A, A3, A4)
    B <- c(B, 0, 0)
  }

  if((census.changes == "enriched" & nr0 < nr & nc0 < nc) |
     (census.changes == "gold")
  )
  {
    A3 <- c(rep(0, nr * nc - 1), 1) # Matrix for p_R,C = 0 constraint
    A4 <- c(rep(0, nr * nc - 2), 1, 0) # Matrix for p_R-1,C = 0 constraint
    A5 <- c(rep(0, nr * (nc - 1) - 1), 1, rep(0, nr)) # Matrix for p_R,C-1 = 0 constraint
    A6 <- c(rep(0, nr * (nc - 1) - 2), 1, 0, rep(0, nr)) # Matrix for p_R-1,C-1 = 0 constraint

    A <- rbind(A, A3, A4, A5, A6)
    B <- c(B, 0, 0, 0, 0)
  }

  return(list("A" = A, "B" = B))
}

##------------------------------------------------

# Function to estimate initial transition matrix for a unit given a global solution
unit_estimate <- function(m0, xi, yi, nr0, nc0, census.changes,
                          eps = 1e-4, ilack.max = 1,
                          kkt2.check = FALSE, trace = FALSE){

  # Preprocessing
  m1 <- as.vector(m0)
  yi <- yi/sum(yi)
  xi <- xi/sum(xi)
  nr <- length(xi)
  nc <- length(yi)

  # Unequal constraints
  lb <- rep(0, nr*nc)
  ub <- rep(1, nr*nc)
  eval_g_ineq1 <- function(z1){
    return( c(z1, 1 - z1) )
  }

  # Equal constraints
  AB <- unit_constraints(xi = xi, yi = yi, nr0 = nr0,
                         nc0 = nc0, census.changes = census.changes)

  # Equal constraints function
  eval_g_eq1 <- function(z1){
    return ( as.numeric(AB$A %*% z1 - AB$B))
  }

  # objective function
  objfun_unit <- function(z1){
    z1 <- matrix(z1, nrow = nr, byrow = FALSE)
    resid1 <- sum( (z1-m0)^2 )
    return(resid1)
  }

  sol1u <- alabama::auglag(par = m1,
                           fn = objfun_unit,
                           hin = eval_g_ineq1,
                           heq = eval_g_eq1,
                           control.outer = list("eps" = eps,
                                                ilack.max = ilack.max,
                                                trace = trace,
                                                kkt2.check = kkt2.check
                          ),
                          control.optim = list(abstol = 1e-4
                          ))$par
  sol1u[sol1u < 0] <- 0
  sol1u[sol1u >= 1] <- 1
  sol1u <- matrix(sol1u, nrow = nr, byrow = FALSE)
  return(sol1u)
}
##------------------------------------------------

# Function for defining the joint constraints
joint_constraints <- function(x, y, nr0, nc0, census.changes){
  nr <- ncol(x)
  nc <- ncol(y)
  nu <- nrow(x)

  # Initial global constraints
  A1 <- NULL
  for (ii in 1L:nu){
    e <- rep(0, nu)
    e[ii] <- 1
    A1 <- rbind(A1, kronecker(e, x[ii, ]/sum(x[ii, ])))
  }
  A1 <- kronecker(diag(nc), A1) # Column sum constraints
  A2 <- kronecker(t(rep(1, nc)), diag(nu * nr)) # unitary-row sum constraints
  A <- rbind(A1, A2)
  # Constraints of sum across rows by units (first set of constraints)
  y.constraints <- as.vector(y/rowSums(y))
  B <- c(y.constraints, rep(1, nu*nr))


  if((census.changes == "raw" & nr0 < nr & nc0 < nc) |
     (census.changes == "regular" & nr0 == nr & nc0 < nc) |
     (census.changes == "ordinary" & nr0 < nr & nc0 == nc) |
     (census.changes == "enriched" & nr0 == nr & nc0 == nc) |
     (census.changes == "semifull")
  )
  {
    # Matrix for p_R,C = 0 constraint for all units
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A3 <- c( rep(0, nr * nu* (nc - 1)), rep(c(rep(0, nr - 1), 1), nu) )
    A <- rbind(A, A3)
    B <- c(B, 0)
  }


  if((census.changes == "regular" & nr0 < nr & nc0 < nc) |
     (census.changes == "enriched" & nr0 < nr & nc0 == nc) |
     (census.changes == "full")
  )
  {
    # Matrix for p_R,C = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A3 <- c( rep(0, nr * nu* (nc - 1)), rep(c(rep(0, nr - 1), 1), nu) )
    # Matrix for p_R-1,C = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A4 <- c( rep(0, nr * nu * (nc - 1)), rep(c(rep(0, nr - 2), 1, 0), nu) )
    A <- rbind(A, A3, A4)
    B <- c(B, 0, 0)
  }

  if((census.changes == "ordinary" & nr0 < nr & nc0 < nc) |
     (census.changes == "enriched" & nr0 == nr & nc0 < nc) |
     (census.changes == "fullreverse")
  )
  {
    # Matrix for p_R,C = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A3 <- c( rep(0, nr * nu* (nc - 1)), rep(c(rep(0, nr - 1), 1), nu) )
    # Matrix for p_R,C-1 = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A4 <- c( rep(0, nr * nu* (nc - 2)), rep(c(rep(0, nr - 1), 1), nu),
             rep(0, nr * nu))
    A <- rbind(A, A3, A4)
    B <- c(B, 0, 0)
  }

  if((census.changes == "enriched" & nr0 < nr & nc0 < nc) |
     (census.changes == "gold")
  )
  {
    # Matrix for p_R,C = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A3 <- c( rep(0, nr * nu* (nc - 1)), rep(c(rep(0, nr - 1), 1), nu) )
    # Matrix for p_R-1,C = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A4 <- c( rep(0, nr * nu * (nc - 1)), rep(c(rep(0, nr - 2), 1, 0), nu) )
    # Matrix for p_R,C-1 = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A5 <- c( rep(0, nr * nu* (nc - 2)), rep(c(rep(0, nr - 1), 1), nu),
             rep(0, nr * nu))
    # Matrix for p_R-1,C-1 = 0 constraint
    # Instead of a constraint of being zero per coefficient,
    # It is stated that its sum must be zero, with all of them non-negative
    A6 <- c( rep(0, nr * nu* (nc - 2)), rep(c(rep(0, nr - 2), 1, 0), nu),
             rep(0, nr * nu))
    A <- rbind(A, A3, A4, A5, A6)
    B <- c(B, 0, 0, 0, 0)
  }

  return(list("A" = A, "B" = B))
}

##-------------------------------------------------------

# This function defines the objective function corresponding to a column of all
# the transfer matrices of all the units
# incog: vector of order N*R of unit proportions defined by R-blocks: p_ir
#        the first R components correspond to the first unit table, the second
#        block of R components to the second unit table and so on, with the last
#        block of R components corresponding to the Nth unit table.
# weights: matrix of order NxR with the weights to be assigned to each discrepancy
f_obj_col <- function(incog, weights){

  incog.m <- matrix(incog, nrow = nrow(weights), byrow = TRUE)
  pesos <- t(t(weights) / rowSums(t(weights)))
  medias <- colSums(pesos * incog.m)
  resid <- sum( pesos * t(t(incog.m) - medias)^2)

}
##-------------------------------------------------------

# Objetive function for the full problem
# This function defines the objective function corresponding to all the columns
# of the transfer matrices of all the units
# incog.t: vector of order N*R*C of unit proportions with C blocks of length N*R,
#          with N*R blocks defined as in incog of function f_obj_col_2
f_obj_tot <- function(incog.t , weights){

  incog.t.m <- matrix(incog.t,
                      nrow = prod(dim(weights)),
                      byrow = FALSE)
  nc <- length(incog.t)/nrow(incog.t.m)
  resid <- 0
  for (jj in 1L:nc){
    resid <- resid + f_obj_col(incog.t.m[, jj], weights)
  }
  return(resid)
}
##-------------------------------------------------------

# Gradient function of the objective function
grad_f <- function(incog.t , weights){
  nr <- ncol(weights) # Number of rows
  nu <- nrow(weights) # Number of units
  pesos <- t(t(weights) / rowSums(t(weights)))
  pesos.1 <- as.vector(t(pesos))
  nc <- length(incog.t)/(nr*nu)
  pesos.1 <- rep(pesos.1 - pesos.1^2, nc)
  comp1 <- 2*incog.t*pesos.1
  comp2 <- NULL
  for (cc in 1L:nc){
    for (uu in 1L:nu){
      for (rr in 1L:nr){
        peso.r <- pesos[, rr]
        peso.u <- peso.r[uu]
        peso.r[uu] <- 0
        incog.t.m <- matrix(incog.t,
                            nrow = nr*nu,
                            byrow = FALSE)[, cc]
        incog.t.m <- matrix(incog.t.m,
                            nrow = nu,
                            byrow = TRUE)[, rr]
        comp2 <- c(comp2, peso.u*sum(peso.r*incog.t.m))
      }
    }
  }
  comp <- comp1 - 2*comp2
  return(comp)
}
##-------------------------------------------------------


