#' Make a list of expressions
#'
#' Constructs a list of expressions, with support for `elixir`'s special
#' [expression][elixir-expression] syntax (expression literals with `{}` or
#' `~{}`, and alternatives with `?`).
#'
#' Be aware that using the `[[` indexing operator on an object of class
#' `expr_list` discards information about whether that element of the list is
#' marked as anchored. In other words, if `xl <- expr_list({.A}, ~{.A})`,
#' then `xl[[1]]` and `xl[[2]]` are both equal to the "bare" symbol `.A`, so
#' the information that the second element of the list is anchored has been
#' lost. Consequently, in e.g. `expr_match(expr, xl[[2]])`, it will be as
#' though the tilde isn't there, and `xl[[2]]` will not just match with the top
#' level of `expr` as was probably intended. Use the `[` operator instead,
#' which retains anchoring information; `expr_match(expr, xl[2])` will work as
#' expected.
#'
#' Note that when you replace part of an `expr_list` with another `expr_list`,
#' the anchoring information from the "replacement" `expr_list` is copied over,
#' while replacing part of an `expr_list` with an expression or a "plain" list
#' of expressions retains the existing anchoring information.
#'
#' @param ... [Expressions][elixir-expression] to include in the list. If the
#' arguments are named, these will be passed on to the returned list.
#' @param env Environment for injections in `...` (see
#' [expression][elixir-expression]).
#' @param xl An `expr_list`.
#' @param i Index for subsetting the `expr_list`; an integer, numeric, logical,
#' or character vector (for named `expr_list`s) interpreted in the usual R way.
#' @param value Replacement; an `expr_list`, an expression, or a list of
#' expressions.
#' @return A list of expressions, of class `expr_list`.
#' @examples
#' expr_list(
#'    ~{ 1 + 1 = 2 } ? ~{ 2 + 2 = 4 },
#'    ~{ y = a * x + b },
#'    { .A }
#' )
#'
#' # There is support for rlang's injection operators.
#' var = as.name("myvar")
#' expr_list({ 1 }, { !!var }, { (!!var)^2 })
#' @order 1
#' @export
expr_list = function(..., env = parent.frame())
{
    # Special case for 0 length list
    if (...length() == 0) {
        return (structure(list(), class = "expr_list", into = logical(0)))
    }

    # Get expressions from ...
    args = substitute(a(...));
    expr = list();
    for (i in seq_len(...length())) {
        expr[[i]] = do.call(do_parse, list(args[[i + 1]], env));
    }

    names(expr) = ...names()
    into = c(unlist(sapply(expr, attr, "into")))
    expr = structure(unlist(expr, recursive = FALSE), class = "expr_list", into = into)

    return (expr)
}

#' @rdname expr_list
#' @order 2
#' @export
`[.expr_list` = function(xl, i)
{
    if (missing(i)) {
        return (xl)
    }
    into = attr(xl, "into")[i]
    into[is.na(into)] = FALSE # For case where any(i > length(xl))
    structure(`[.simple.list`(xl, i), class = "expr_list", into = into)
}

rep_len_warn = function(x, length.out)
{
    if (length.out %% length(x) != 0) {
        warning("number of items to replace is not a multiple of replacement length", call. = FALSE)
    }
    rep_len(x, length.out)
}

index_replace = function(xl, i, value, cl)
{
    if (is.null(value)) {
        # value is NULL: omit elements in i
        i = seq_along(xl)[i] # Work with positive, valid subscripts
        i = i[!is.na(i)]
        if (length(i) > 0) {
            return (structure(`[.simple.list`(xl, -i), class = cl, into = attr(xl, "into")[-i]))
        } else {
            return (xl)
        }
    }

    # To avoid recursing into this function below
    l = unclass(xl);
    into = attr(xl, "into");

    # Expand length of list, if needed
    if (any(i > length(xl))) {
        l[i[i > length(xl)]] = list(NULL)
        into[i[i > length(xl)]] = FALSE
    }

    nreplace = length(into[i]);

    if (inherits(value, "expr_list")) {
        l[i] = rep_len_warn(value, nreplace);
        into[i] = rep_len(attr(value, "into"), nreplace);
    } else if (is.list(value)) {
        if (any(!sapply(value, function(x) { rlang::is_expression(x) || class(x) == "expr_alt" }))) {
            stop("When replacing part of an ", cl, " with a list, each element of that list must be a valid expr_list element.")
        }
        l[i] = rep_len_warn(value, nreplace);
    } else if (rlang::is_expression(value)) {
        l[i] = rep_len_warn(list(value), nreplace);
    } else {
        stop("Can only replace part of an ", cl, " with an expr_list, a list of expressions, or an expression.")
    }
    structure(l, class = cl, into = into);
}

#' @rdname expr_list
#' @order 3
#' @export
`[<-.expr_list` = function(xl, i, value)
{
    index_replace(xl, i, value, "expr_list")
}

#' @keywords internal
#' @export
print.expr_list = function(x, ...)
{
    if (length(x) != length(attr(x, "into"))) {
        warning("Corrupted expr_list: length of list not equal to length of 'into' attribute.")
    }

    fmt_expr = function(expr, unanchored) {
        paste(if (unanchored) '{' else '~{', format(expr), '}')
    }

    str = mapply(
        function(y, i, n) {
            if (n != "") {
                n = paste0(n, " = ")
            }
            if (inherits(y, "expr_alt")) {
                return (paste0(n, paste0(mapply(fmt_expr, y, attr(y, "into")), collapse = " ? ")))
            } else {
                return (paste0(n, fmt_expr(y, i)))
            }
        },
        x, attr(x, "into"), rlang::names2(x))

    cat("expr_list of length ", length(x), ": ", paste0(str, collapse = ", "), sep = "")
}

#' Assign to part of an `expr_alt`.
#'
#' This exists primarily so that `expr_apply` can be applied to an `expr_list`,
#' which may potentially contain elements of class `expr_alt`.
#'
#' @return The modified object of class `"expr_alt"`.
#' @keywords internal
#' @export
`[<-.expr_alt` = function(xl, i, value)
{
    index_replace(xl, i, value, "expr_alt")
}

#' Get or set a subexpression
#'
#' These functions allow you to extract and/or modify a subexpression within an
#' expression.
#'
#' The `elixir` functions [expr_match()] and [expr_locate()] find matching
#' "subexpressions" within expressions and return indices that allow accessing
#' these subexpressions. For example, the expression `1 + 2 + 3` contains all
#' the following subexpressions:
#'
#' | **index** | **subexpression** | **accessed with R code**              |
#' |:--------- |:-----------------:|:------------------------------------- |
#' | `NULL`    | `1+2+3`           | `expr`                                |
#' | `1`       | `+`               | `expr[[1]]`                           |
#' | `2`       | `1+2`             | `expr[[2]]`                           |
#' | `3`       | `3`               | `expr[[3]]`                           |
#' | `c(2,1)`  | `+`               | `expr[[2]][[1]]` or `expr[[c(2, 1)]]` |
#' | `c(2,2)`  | `1`               | `expr[[2]][[2]]` or `expr[[c(2, 2)]]` |
#' | `c(2,3)`  | `2`               | `expr[[2]][[3]]` or `expr[[c(2, 3)]]` |
#'
#' Any index returned by [expr_match()] or [expr_locate()] will either be
#' `NULL` (meaning the whole expression / expression list) or an integer vector
#' (e.g. `1` or `c(2,3)` in the table above).
#'
#' Suppose you have an index, `idx`. If `idx` is an integer vector, you can
#' just use `expr[[idx]]` to access the subexpression. But in the case where
#' `idx` is `NULL`, R will complain that you are trying to select less than one
#' element. The sole purpose of [expr_sub()] is to get around that issue and
#' allow you to pass either `NULL` or an integer vector as the index you want
#' for an expression or list of expressions.
#'
#' @param expr The expression to select from. Can also be a list of
#'     expressions, in which case the first element of `index` selects the
#'     expression from the list. Can also be a formula.
#' @param idx A valid index: `NULL` or an integer vector.
#' @param env Environment for any injections in `expr` (see
#' [expression][elixir-expression]).
#' @param value Replacement; an expression.
#' @return The element of the expression selected by `idx`.
#' @seealso [expr_match()], [expr_locate()] which return indices to
#' subexpressions.
#' @examples
#' expr = quote(y == a * x + b)
#' expr_sub(expr, NULL)
#' expr_sub(expr, 3)
#' expr_sub(expr, c(3, 3))
#'
#' expr_sub(expr, c(3, 3)) <- quote(q)
#' print(expr)
#' @export
expr_sub = function(expr, idx, env = parent.frame())
{
    expr = do.call(do_parse_simple, list(substitute(expr), env))
    if (inherits(expr, "expr_wrap")) {
        expr = expr[[1]]
    }

    if (length(idx) == 0) {
        return (expr)
    } else {
        return (expr[[idx]])
    }
}

#' @rdname expr_sub
#' @export
`expr_sub<-` = function(expr, idx, env = parent.frame(), value)
{
    expr = do.call(do_parse_simple, list(substitute(expr), env))
    if (inherits(expr, "expr_wrap")) {
        expr = expr[[1]]
    }

    if (length(idx) == 0) {
        return (value)
    } else {
        expr[[idx]] = value;
        return (expr)
    }
}

# Parses expressions passed to the expr_* functions with brace-wrapped
# expression literals, also including information on how to return the result
# in the "original" format.
do_parse_simple = function(expr, env = parent.frame())
{
    # Unevaluated expr as list
    lx = as.list(substitute(expr));

    # Special case to handle passing NULL
    if (length(lx) == 0) {
        return (structure(list(NULL), class = "expr_wrap", into = NA))
    }

    # Check for mistakes in specifying -- tilde or question mark
    if (length(lx) > 1) {
        if (length(lx) == 2 && identical(lx[[1]], quote(`~`)) && length(lx[[2]]) > 1 && identical(lx[[2]][[1]], quote(`{`))) {
            stop("Do not use anchor operator (~) in first argument.", call. = FALSE)
        }

        if (length(lx) > 0 && identical(lx[[1]], quote(`?`))) {
            stop("Do not use alternatives operator (?) in first argument.", call. = FALSE)
        }
    }

    # Convert env to an environment if it is a list
    if (is.list(env)) {
        env = list2env(env)
    }

    # Parse argument
    lx = debrace(lx, substitute(expr), env)
    if (is.list(lx)) {
        # Already list
        return (lx)
    }

    return (structure(list(lx), class = "expr_wrap", into = NA))
}

# Parses pattern/replacement expressions passed to the expr_* functions with a
# special syntax including brace-wrapped expression literals, ? for enumerating
# alternatives and ~ for specifying that the expression is a pattern that is
# anchored at the top level.
do_parse = function(expr, env = parent.frame())
{
    # Unevaluated expr as list
    lx = as.list(substitute(expr));

    # Special cases needed to handle passing NULL or ~NULL
    if (length(lx) == 0) {
        return (structure(list(NULL), class = "expr_list", into = TRUE))
    } else if (identical(lx, list(quote(`~`), NULL))) {
        return (structure(list(NULL), class = "expr_list", into = FALSE))
    }

    # Convert env to an environment if it is a list
    if (is.list(env)) {
        env = list2env(env)
    }

    # Parse argument
    # Handle alternatives (`?` notation)
    if (identical(lx[[1]], quote(`?`))) {
        if (length(lx) == 2 && !(is.call(lx[[2]]) && identical(lx[[2]][[1]], quote(`{`)))) {
            # Format: ?z or ?~z, where z is something evaluating to a list, but is NOT brace wrapped
            if (is.call(lx[[2]]) && identical(lx[[2]][[1]], quote(`~`)) && length(lx[[2]]) == 2) {
                alts = eval(lx[[2]][[2]], env)
                into = FALSE;
            } else {
                alts = eval(lx[[2]], env)
                into = TRUE;
            }
            if (!is.list(alts)) {
                stop("In ?x or ?~x, x must evaluate to a list.")
            }
            # This can be returned directly with no further processing
            return (structure(list(structure(alts, class = "expr_alt", into = rep(into, length(alts)))),
                class = "expr_list", into = NA))
        } else if (length(lx) == 3) {
            # Format: a ? b [? c ...]
            result = structure(dequestion(lx), class = "expr_alt")
            to_eval = dequestion(substitute(expr))

            result = list(detilde(result))
            to_eval = list(detilde(to_eval, TRUE))
            attr(result, "into") = NA
        } else {
            stop("Could not parse ", lang2str(substitute(expr)))
        }
    } else {
        # Single alternative (no ?) -- one-element list
        result = detilde(list(lx))
        to_eval = detilde(list(substitute(expr)), TRUE)
    }

    # Process the list
    for (i in seq_along(result)) {
        if (inherits(result[[i]], "expr_alt")) {
            for (j in seq_along(result[[i]])) {
                result[[i]][j] = list(debrace(result[[i]][[j]], to_eval[[i]][[j]], env))
            }
        } else {
            result[i] = list(debrace(result[[i]], to_eval[[i]], env))
            if (inherits(result[[i]], "expr_list")) {
                # Already expr_list
                return (result[[i]])
            } else if (is.list(result[[i]])) {
                # Evaluates to a list -- treat this as the result in itself
                result = structure(result[[i]], into = rep(attr(result, "into"), length(result[[i]])))
                break
            }
        }
    }

    # Set attributes and return; this particular resetting of both class and
    # into is to ensure order of these two attributes is the same, which is not
    # strictly required but may be good for consistency's sake.
    attributes(result) = list(class = "expr_list", into = attr(result, "into"))
    return (result)
}

# Removes tildes and stores their presence as TRUE in attribute "into"
detilde = function(x, to_eval = FALSE)
{
    into = rep(TRUE, length(x))
    for (i in seq_along(x)) {
        if (!is.name(x[[i]]) && identical(x[[i]][[1]], quote(`~`))) {
            into[i] = FALSE
            if (to_eval) {
                x[[i]] = x[[i]][[2]]
            } else {
                x[[i]] = as.list(x[[i]][[2]])
            }
        }
    }
    structure(x, into = into)
}

# Unwraps a ? b [ ? c [ ? d ... ]] into a list
dequestion = function(x)
{
    if (is.call(x[[2]]) && identical(x[[2]][[1]], quote(`?`))) {
        head = dequestion(x[[2]])
    } else {
        head = list(x[[2]])
    }
    return (c(head, list(x[[3]])))
}

# If x is brace-wrapped, remove braces and inject, otherwise return ev
debrace = function(x, ev, env)
{
    if ((is.list(x) || is.call(x)) && identical(x[[1]], quote(`{`))) {
        # If element is { }-wrapped, treat as quotation; debrace and inject
        if (length(x) != 2) {
            stop("Contents of { } must be a single expression.")
        } else {
            return (rlang::inject(rlang::expr(!!x[[2]]), env))
        }
    } else {
        # Otherwise provide the evaluation of expr in the parent frame
        return (eval(ev, env))
    }
}

# Returns TRUE if x is an expression (rlang::expression()) or a formula, i.e.
# tests whether x is suitable as the 1st expr argument to the expr_ functions.
is_expr1 = function(x)
{
    rlang::is_expression(x) || rlang::is_formula(x, scoped = TRUE)
}

#' Expressions in `elixir`
#'
#' @description `elixir` is primarily a package for working with what it calls
#' "expressions", in the sense of any R object for which
#' [rlang::is_expression()] returns `TRUE`. This includes calls, like the
#' results of evaluating `quote(f(x))` or `quote(a:b)`, symbols like
#' `quote(z)`, and syntactic literals like `2.5`, `"hello"`, `NULL`, `FALSE`,
#' and so on. In many cases, you can also use `elixir` to work with
#' [formulas][base::tilde], even though [rlang::is_expression()] returns
#' `FALSE` for formulas.
#'
#' This is not to be confused with the built-in type [base::expression], which
#' is essentially a special way of storing a vector of multiple "expressions".
#' `elixir` does not use this type; see [expr_list()] instead.
#'
#' @section Usage:
#' ```
#' expr_list(number = { `.A:numeric` } ? { `.A:integer` },
#'     string = { `.A:character` }, symbol = { `.A:name` })
#' expr_match({ 1 * 2 }, ~{ .A * .B })
#' expr_match({ 1 * 2 }, { `.A:numeric` })
#' expr_replace({ y = a*x^3 + b*x^2 + c*x^1 + d*x^0 },
#'     { ..X ^ ..N }, { pow(..X, ..N) })
#' ```
#'
#' @section Specifying expressions in `elixir`:
#'
#' The `elixir` package functions starting with `expr_` work with expressions.
#' These functions all accept a special (optional) syntax for specifying
#' expressions which involves the symbols `{}`, `?`, and `~`, as well as the
#' rlang [injection operator, !!][rlang::injection-operator] and
#' [splice operator, !!!][rlang::splice-operator]).
#'
#' With base R, if you want to store an expression such as `x + y` in a
#' variable or pass it to a function, you need to use [base::quote()] or
#' [rlang::expr()], but any Elixir `expr_` function will also accept an
#' "expression literal" wrapped in braces, `{}`.
#'
#' So, for example, rather than
#'
#' `translate(quote(x ^ y), "C++")`
#'
#' you can write
#'
#' `translate({ x ^ y }, "C++")`.
#'
#' This only works if the braces are provided "directly"; that is, in
#'
#' `expr <- quote({ x ^ y }); translate(expr, "C++")`,
#'
#' the braces are not interpreted in any special way.
#'
#' Anything between the braces essentially gets put through [rlang::expr()], so
#' you can use `!!` (i.e. [rlang::injection-operator]) and `!!!` (i.e.
#' [rlang::splice-operator]). There is an `env` parameter to all relevant
#' `elixir` functions, defaulting to `parent.frame()`, in which these injection
#' operations are evaluated.
#'
#' @section Special syntax for patterns and replacements:
#'
#' Additionally, some functions ([expr_match()], [expr_count()],
#' [expr_detect()], [expr_extract()], [expr_locate()], and [expr_replace()])
#' take `pattern` and/or `replacement` arguments to specify patterns to match
#' to an expression and/or replacement expressions to replace those matches
#' with.
#'
#' For both `pattern` and `replacement` arguments, you can use the question
#' mark operator `?` to specify *alternatives*. For example, to match *either*
#' the token `cat` or `dog`, you can use
#'
#' `expr_match(expr, { cat } ? { dog })`.
#'
#' You can chain together as many alternatives as are needed. Alternatively,
#' if you have a list of expressions `z`, you can use a single question mark
#' before the name of the list, like so:
#'
#' `expr_match(expr, ?z)`
#'
#' and `elixir` will treat the list as a set of alternatives. When using
#' [expr_replace()] with a set of alternatives as the pattern, the replacement
#' needs to be either a single expression, or a set of alternative expressions
#' which has the same number of alternatives as in the pattern.
#'
#' You can also use the tilde operator `~` to specify that a given pattern
#' should be "anchored" at the top level of an expression, and will not
#' "recurse into" the expression. For example, in
#'
#' ```
#' exprs = expr_list(2, 5, {1 + 4})
#' expr_match(exprs, ~{ `.A:numeric` })
#' ```
#'
#' only the numbers `2` and `5` will match. However, in
#'
#' ```
#' exprs = expr_list(2, 5, {1 + 4})
#' expr_match(exprs, { `.A:numeric` })
#' ```
#'
#' all numbers `2`, `5`, `1` and `4` will match, because the `pattern` can
#' recurse into the third expression `1 + 4`.
#'
#' @name elixir-expression
NULL

#' `elixir`: Transmutation of languages
#'
#' `elixir` is a set of tools for transforming R expressions, including
#' into other programming languages.
#'
#' One of the neat features of R is that you can use the language to
#' inspect itself. Expressions, functions, indeed entire R scripts can be
#' examined and manipulated just like any list, data.frame, or other R
#' object.
#'
#' However, the syntax for manipulating R language objects is a little
#' tricky. Packages such as `rlang` help to make this task easier. `elixir`
#' makes a few extra shortcuts available, and is geared for advanced R
#' users.
#'
#' `elixir` provides functions for finding, extracting, and replacing patterns
#' in 'R' language objects, similarly to how regular expressions can be used to
#' find, extract, and replace patterns in text. It also provides functions for
#' generating code using specially-formatted template files and for translating
#' 'R' expressions into similar expressions in other programming languages.
#'
#' The package may be helpful for advanced uses of 'R' expressions, such as
#' developing domain-specific languages.
#'
#' @section Find and replace for language objects:
#'
#' Sometimes you want to detect certain patterns within an expression or
#' list of expressions, or easily replace a certain pattern with another.
#' When working with strings, regular expressions are a handy way of
#' accomplishing such tasks. `elixir` provides a sort of "regular
#' expressions for R expressions" functionality through the functions
#' [expr_match()], [expr_replace()], and the "shortcut" functions
#' [expr_count()], [expr_detect()], [expr_extract()], and [expr_locate()].
#'
#' @section Other `elixir` features:
#'
#' The function [expr_apply()] allows you to transform and extract information
#' from nested list structures which contain expressions, so if you have a big
#' structure and you want to check all the variable names or make certain
#' replacements, this may be useful.
#'
#' [expr_sub()] offers an interface for extracting or replacing part of an
#' expression; the one advantage this has over `[[` is that it allows you to use
#' `NULL` as the index, which gives back the whole expression.
#'
#' [lang2str()] does the opposite of [base::str2lang()]; it is like
#' `deparse1()` which is new since R 4.0.0, but with `collapse = ""` instead of
#' `collapse = " "`.
#'
#' Finally, [meld()], [translate()], and [reindent()] are various experimental
#' functions for constructing code using R.
#'
#' @name elixir
NULL
