% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/r0_conversions.R
\name{r0_conversions}
\alias{r0_conversions}
\alias{lambda_to_r0}
\alias{r0_to_lambda}
\alias{r_eff}
\title{Converts between epidemiological parameters related to \eqn{R_0}}
\usage{
lambda_to_r0(
  lambda,
  contact_matrix,
  demography_vector,
  susceptibility,
  p_susceptibility,
  infectious_period = 1.8
)

r0_to_lambda(
  r0,
  contact_matrix,
  demography_vector,
  susceptibility,
  p_susceptibility,
  infectious_period = 1.8
)

r_eff(r0, contact_matrix, demography_vector, susceptibility, p_susceptibility)
}
\arguments{
\item{lambda}{The transmission rate of the disease, also called the 'force of
infection' (\eqn{\lambda}). This is different from the effective
transmission rate (\eqn{\beta}).}

\item{contact_matrix}{Social contact matrix. Entry \eqn{m_{ij}} gives average
number of contacts in group \eqn{i} reported by participants in group \eqn{j}}

\item{demography_vector}{Demography vector. Entry \eqn{v_{i}} gives
proportion of total population in group \eqn{i}.}

\item{susceptibility}{A matrix giving the susceptibility of individuals in
demographic group \eqn{i} and risk group \eqn{k}.}

\item{p_susceptibility}{A matrix giving the probability that an individual
in demography group \eqn{i} is in risk (or susceptibility) group \eqn{k}.
Each row represents the overall distribution of individuals in demographic
group \eqn{i} across risk groups, and each row \emph{must sum to 1.0}.}

\item{infectious_period}{Duration of the infectious period in days.
Default value is 1.8 days.}

\item{r0}{The basic reproductive number \eqn{R_0} of the infection.}
}
\value{
Returns a single number for the calculated value.
}
\description{
Converts between \eqn{R_0} and the transmission rate
\eqn{\lambda}, or calculates
the effective reproduction number \eqn{R_\text{eff}} for a population,
while accounting for population characteristics including demographic
heterogeneity in social contacts, heterogeneity in the demographic
distribution, and heterogeneity in susceptibility to infection.

Uses the R0 (\eqn{R_0}), contact matrix (\eqn{C}),
population (\eqn{N}), and infectious period (\eqn{\gamma})
to calculate the transmission rate using the following equation.
\deqn{\lambda = R_0 / ({Max}(EV(C)) \gamma)}
where \eqn{EV(C)} denotes the eigenvalues of the matrix \eqn{C} which is
calculated from the social contacts matrix scaled by the number of
individuals in each demographic and susceptibility group in the population.
}
\details{
Given the transmission rate (\eqn{\lambda}),
social contacts matrix (\eqn{c}), demography (\eqn{N}), the
distribution \eqn{P} of each demographic group \eqn{i} into
susceptibility groups \eqn{S}, and the infectious period (\eqn{\gamma})
\itemize{
\item \code{r_eff()} calculates the effective reproductive number;
\item \code{lamda_to_r0()} calculates the \eqn{R_0} from the transmission rate as
\deqn{R_0 = {Max}(EV(C)) \times \lambda \gamma}
\item \code{r0_to_lambda()} calculates the transmission rate as
\deqn{\lambda = R_0 / ({Max}(EV(C)) \gamma)}
Note that this is also called the 'force of infection' and is different from
the effective transmission rate often denoted \eqn{\beta}.
}

Here, \eqn{EV(C)} denotes the eigenvalues of the matrix \eqn{C} which is
calculated from the social contacts matrix scaled by the number of
individuals in each demographic and susceptibility group in the population.
}
\examples{
#### Prepare data ####
# Get example dataset and prepare contact matrix and demography
data(polymod_uk)
contact_matrix <- polymod_uk$contact_matrix
demography_vector <- polymod_uk$demography_vector

# define lambda
lambda <- 0.3

# define infectious period of 5 days
infectious_period <- 5
# define the number of age and susceptibility groups
n_demo_grps <- length(demography_vector)
n_risk_grps <- 3

# In this example, risk varies across groups
susceptibility <- matrix(
  data = c(0.5, 0.7, 1.0), nrow = n_demo_grps, ncol = n_risk_grps
)

# risk does not vary within groups
p_susceptibility <- matrix(
  data = 1, nrow = n_demo_grps, ncol = n_risk_grps
)
# p_susceptibility rows must sum to 1.0
p_susceptibility <- p_susceptibility / rowSums(p_susceptibility)

#### Effective R ####
r0 <- 2.0
r_eff(
  r0 = r0,
  contact_matrix = contact_matrix,
  demography_vector = demography_vector,
  susceptibility = susceptibility,
  p_susceptibility = p_susceptibility
)

#### Transmission rate to R0 ####
lambda_to_r0(
  lambda, contact_matrix, demography_vector,
  susceptibility, p_susceptibility,
  infectious_period
)

#### R0 to Transmission rate ####
r0 <- 1.5
r0_to_lambda(
  r0, contact_matrix, demography_vector,
  susceptibility, p_susceptibility,
  infectious_period
)
}
