% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_mvt.R
\name{fit_mvt}
\alias{fit_mvt}
\title{Estimate parameters of a multivariate Student's t distribution to fit data}
\usage{
fit_mvt(
  X,
  na_rm = TRUE,
  nu = c("iterative", "kurtosis", "MLE-diag", "MLE-diag-resampled", "cross-cumulants",
    "all-cumulants", "Hill"),
  nu_iterative_method = c("POP", "OPP", "OPP-harmonic", "ECME", "ECM", "POP-approx-1",
    "POP-approx-2", "POP-approx-3", "POP-approx-4", "POP-exact", "POP-sigma-corrected",
    "POP-sigma-corrected-true"),
  initial = NULL,
  optimize_mu = TRUE,
  weights = NULL,
  scale_covmat = FALSE,
  PX_EM_acceleration = TRUE,
  nu_update_start_at_iter = 1,
  nu_update_every_num_iter = 1,
  factors = ncol(X),
  max_iter = 100,
  ptol = 0.001,
  ftol = Inf,
  return_iterates = FALSE,
  verbose = FALSE
)
}
\arguments{
\item{X}{Data matrix containing the multivariate time series (each column is one time series).}

\item{na_rm}{Logical value indicating whether to remove observations with some NAs (default is \code{TRUE}).
Otherwise, the NAs will be imputed at a higher computational cost.}

\item{nu}{Degrees of freedom of the \eqn{t} distribution. Either a number (\code{>2}) or a string indicating the
method to compute it:
\itemize{\item{\code{"iterative"}: iterative estimation (with method to be specified in argument
                                   \code{nu_iterative_method}) with the rest of the parameters (default method);}
         \item{\code{"kurtosis"}: one-shot estimation based on the kurtosis obtained from the sampled moments;}
         \item{\code{"MLE-diag"}: one-shot estimation based on the MLE assuming a diagonal sample covariance;}
         \item{\code{"MLE-diag-resampled"}: like method \code{"MLE-diag"} but resampled for better stability.}}}

\item{nu_iterative_method}{String indicating the method for iteratively estimating \code{nu} (in case \code{nu = "iterative"}):
\itemize{\item{\code{"ECM"}: maximization of the Q function [Liu-Rubin, 1995];}
         \item{\code{"ECME"}: maximization of the log-likelihood function [Liu-Rubin, 1995];}
         \item{\code{"OPP"}: estimator from paper [Ollila-Palomar-Pascal, TSP2021, Alg. 1];}
         \item{\code{"OPP-harmonic"}: variation of \code{"OPP"};}
         \item{\code{"POP"}: improved estimator as in paper
                             [Pascal-Ollila-Palomar, EUSIPCO2021, Alg. 1] (default method).}}}

\item{initial}{List of initial values of the parameters for the iterative estimation method (in case \code{nu = "iterative"}).
Possible elements include:
\itemize{\item{\code{mu}: default is the data sample mean,}
         \item{\code{cov}: default is the data sample covariance matrix,}
         \item{\code{scatter}: default follows from the scaled sample covariance matrix,}
         \item{\code{nu}: can take the same values as argument \code{nu}, default is \code{4},}
         \item{\code{B}: default is the top eigenvectors of \code{initial$cov}
                                   multiplied by the sqrt of the eigenvalues,}
         \item{\code{psi}: default is
                          \code{diag(initial$cov - initial$B \%*\% t(initial$B)).}}}}

\item{optimize_mu}{Boolean indicating whether to optimize \code{mu} (default is \code{TRUE}).}

\item{weights}{Optional weights for each of the observations (the length should be equal to the number of rows of X).}

\item{scale_covmat}{Logical value indicating whether to scale the scatter and covariance matrices to minimize the MSE
estimation error by introducing bias (default is \code{FALSE}). This is particularly advantageous
when the number of observations is small compared to the number of variables.}

\item{PX_EM_acceleration}{Logical value indicating whether to accelerate the iterative method via
the PX-EM acceleration technique (default is \code{TRUE}) [Liu-Rubin-Wu, 1998].}

\item{nu_update_start_at_iter}{Starting iteration (default is 1) for
iteratively estimating \code{nu} (in case \code{nu = "iterative"}).}

\item{nu_update_every_num_iter}{Frequency (default is 1) for
iteratively estimating \code{nu} (in case \code{nu = "iterative"}).}

\item{factors}{Integer indicating number of factors (default is \code{ncol(X)}, so no factor model assumption).}

\item{max_iter}{Integer indicating the maximum number of iterations for the iterative estimation
method (default is \code{100}).}

\item{ptol}{Positive number indicating the relative tolerance for the change of the variables
to determine convergence of the iterative method (default is \code{1e-3}).}

\item{ftol}{Positive number indicating the relative tolerance for the change of the log-likelihood
value to determine convergence of the iterative method (default is \code{Inf}, so it is
not active). Note that using this argument might have a computational cost as a convergence
criterion due to the computation of the log-likelihood (especially when \code{X} is high-dimensional).}

\item{return_iterates}{Logical value indicating whether to record the values of the parameters (and possibly the
log-likelihood if \code{ftol < Inf}) at each iteration (default is \code{FALSE}).}

\item{verbose}{Logical value indicating whether to allow the function to print messages (default is \code{FALSE}).}
}
\value{
A list containing (possibly) the following elements:
        \item{\code{mu}}{Mu vector estimate.}
        \item{\code{scatter}}{Scatter matrix estimate.}
        \item{\code{nu}}{Degrees of freedom estimate.}
        \item{\code{mean}}{Mean vector estimate: \preformatted{mean = mu}}
        \item{\code{cov}}{Covariance matrix estimate: \preformatted{cov = nu/(nu-2) * scatter}}

        \item{\code{converged}}{Boolean denoting whether the algorithm has converged (\code{TRUE}) or the maximum number
                                of iterations \code{max_iter} has been reached (\code{FALSE}).}
        \item{\code{num_iterations}}{Number of iterations executed.}
        \item{\code{cpu_time}}{Elapsed CPU time.}
        \item{\code{B}}{Factor model loading matrix estimate according to \code{cov = (B \%*\% t(B) + diag(psi)}
                        (only if factor model requested).}
        \item{\code{psi}}{Factor model idiosynchratic variances estimates according to \code{cov = (B \%*\% t(B) + diag(psi)}
                          (only if factor model requested).}
        \item{\code{log_likelihood_vs_iterations}}{Value of log-likelihood over the iterations (if \code{ftol < Inf}).}
        \item{\code{iterates_record}}{Iterates of the parameters (\code{mu}, \code{scatter}, \code{nu},
                                      and possibly \code{log_likelihood} (if \code{ftol < Inf})) along the iterations
                                      (if \code{return_iterates = TRUE}).}
}
\description{
Estimate parameters of a multivariate Student's t distribution to fit data,
namely, the mean vector, the covariance matrix, the scatter matrix, and the degrees of freedom.
The data can contain missing values denoted by NAs.
It can also consider a factor model structure on the covariance matrix.
The estimation is based on the maximum likelihood estimation (MLE) and the algorithm is
obtained from the expectation-maximization (EM) method.
}
\details{
This function estimates the parameters of a multivariate Student's t distribution (\code{mu},
         \code{cov}, \code{scatter}, and \code{nu}) to fit the data via the expectation-maximization (EM) algorithm.
         The data matrix \code{X} can contain missing values denoted by NAs.
         The estimation of \code{nu} if very flexible: it can be directly passed as an argument (without being estimated),
         it can be estimated with several one-shot methods (namely, \code{"kurtosis"}, \code{"MLE-diag"},
         \code{"MLE-diag-resampled"}), and it can also be iteratively estimated with the other parameters via the EM
         algorithm.
}
\examples{
library(mvtnorm)       # to generate heavy-tailed data
library(fitHeavyTail)

X <- rmvt(n = 1000, df = 6)  # generate Student's t data
fit_mvt(X)

# setting lower limit for nu
options(nu_min = 4.01)
fit_mvt(X, nu = "iterative")

}
\references{
Chuanhai Liu and Donald B. Rubin, "ML estimation of the t-distribution using EM and its extensions, ECM and ECME,"
Statistica Sinica (5), pp. 19-39, 1995.

Chuanhai Liu, Donald B. Rubin, and Ying Nian Wu, "Parameter Expansion to Accelerate EM: The PX-EM Algorithm,"
Biometrika, Vol. 85, No. 4, pp. 755-770, Dec., 1998

Rui Zhou, Junyan Liu, Sandeep Kumar, and Daniel P. Palomar, "Robust factor analysis parameter estimation,"
Lecture Notes in Computer Science (LNCS), 2019. <https://arxiv.org/abs/1909.12530>

Esa Ollila, Daniel P. Palomar, and Frédéric Pascal, "Shrinking the Eigenvalues of M-estimators of Covariance Matrix,"
IEEE Trans. on Signal Processing, vol. 69, pp. 256-269, Jan. 2021. <https://doi.org/10.1109/TSP.2020.3043952>

Frédéric Pascal, Esa Ollila, and Daniel P. Palomar, "Improved estimation of the degree of freedom parameter of
multivariate t-distribution," in Proc. European Signal Processing Conference (EUSIPCO), Dublin, Ireland, Aug. 23-27, 2021.
<https://doi.org/10.23919/EUSIPCO54536.2021.9616162>
}
\seealso{
\code{\link{fit_Tyler}}, \code{\link{fit_Cauchy}}, \code{\link{fit_mvst}},
         \code{\link{nu_OPP_estimator}}, and \code{\link{nu_POP_estimator}}
}
\author{
Daniel P. Palomar and Rui Zhou
}
