% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/forest_model.R
\name{forest_model}
\alias{forest_model}
\title{Produce a forest plot based on a regression model}
\usage{
forest_model(
  model,
  panels = default_forest_panels(model, factor_separate_line = factor_separate_line),
  covariates = NULL,
  exponentiate = NULL,
  funcs = NULL,
  factor_separate_line = FALSE,
  format_options = forest_model_format_options(),
  theme = theme_forest(),
  limits = NULL,
  breaks = NULL,
  return_data = FALSE,
  recalculate_width = TRUE,
  recalculate_height = TRUE,
  model_list = NULL,
  merge_models = FALSE,
  exclude_infinite_cis = TRUE
)
}
\arguments{
\item{model}{regression model produced by \code{\link[stats]{lm}},
\code{\link[stats]{glm}}, \code{\link[survival]{coxph}}}

\item{panels}{\code{list} with details of the panels that make up the plot (See Details)}

\item{covariates}{a character vector optionally listing the variables to include in the plot
(defaults to all variables)}

\item{exponentiate}{whether the numbers on the x scale should be exponentiated for plotting}

\item{funcs}{optional list of functions required for formatting \code{panels$display}}

\item{factor_separate_line}{whether to show the factor variable name on a separate line}

\item{format_options}{formatting options as a list as generated by \code{\link{forest_model_format_options}}}

\item{theme}{theme to apply to the plot}

\item{limits}{limits of the forest plot on the X-axis (taken as the range of the data
by default)}

\item{breaks}{breaks to appear on the X-axis (note these will be exponentiated
if \code{exponentiate == TRUE})}

\item{return_data}{return the data to produce the plot as well as the plot itself}

\item{recalculate_width}{\code{TRUE} to recalculate panel widths using the current device
or the desired plot width in inches}

\item{recalculate_height}{\code{TRUE} to shrink text size using the current device
or the desired plot height in inches}

\item{model_list}{list of models to incorporate into a single forest plot}

\item{merge_models}{if `TRUE`, merge all models in one section.}

\item{exclude_infinite_cis}{whether to exclude points and confidence intervals
that go to positive or negative infinity from plotting. They will still be
displayed as text. Defaults to \code{TRUE}, since otherwise plot is malformed}
}
\value{
A ggplot ready for display or saving, or (with \code{return_data == TRUE},
  a \code{list} with the parameters to call \code{\link{panel_forest_plot}} in the
  element \code{plot_data} and the ggplot itself in the element \code{plot})
}
\description{
Produce a forest plot based on a regression model
}
\details{
This function takes the model output from one of the common model functions in
  R (e.g. \code{\link[stats]{lm}}, \code{\link[stats]{glm}},
 \code{\link[survival]{coxph}}). If a \code{label} attribute was present on any of the
 columns in the original data (e.g. from the \code{labelled} package),
 this label is used in preference to the column name.

 The \code{panels} parameter is a \code{list} of lists each of which have an element
 \code{width}
 and, optionally, \code{item}, \code{display}, \code{display_na},
 \code{heading}, \code{hjust} and \code{fontface}. \code{item} can be \code{"forest"} for the forest
 plot (exactly one required) or \code{"vline"} for a vertical line.
 \code{display} indicates which column to display as text. It can be a quoted variable name
 or a formula. The column display can include the standard ones produced by
 \code{\link[broom]{tidy}} and in addition
 \code{variable} (the term in the model; for factors this is the bare variable without the  level),
 \code{level} (the level of factors),
 \code{reference} (TRUE for the reference level of a factor). For \code{\link[survival]{coxph}}
 models, there will also be \code{n_events} for the number of events in the group with
 that level of the factor and \code{person_time} for the person-time in that group.
 The function \code{trans} is definded to be the
 transformation between the coefficients and the scales (e.g. \code{exp}). Other functions not
 in base R can be provided as a \code{list} with the parameter \code{funcs}.
 \code{display_na} allows for an alternative display for NA terms within \code{estimate}.
}
\examples{

library("survival")
library("dplyr")
pretty_lung <- lung \%>\%
  transmute(time,
    status,
    Age = age,
    Sex = factor(sex, labels = c("Male", "Female")),
    ECOG = factor(lung$ph.ecog),
    `Meal Cal` = meal.cal
  )

print(forest_model(coxph(Surv(time, status) ~ ., pretty_lung)))

# Example with custom panels

panels <- list(
  list(width = 0.03),
  list(width = 0.1, display = ~variable, fontface = "bold", heading = "Variable"),
  list(width = 0.1, display = ~level),
  list(width = 0.05, display = ~n, hjust = 1, heading = "N"),
  list(width = 0.05, display = ~n_events, width = 0.05, hjust = 1, heading = "Events"),
  list(
    width = 0.05,
    display = ~ replace(sprintf("\%0.1f", person_time / 365.25), is.na(person_time), ""),
    heading = "Person-\nYears", hjust = 1
  ),
  list(width = 0.03, item = "vline", hjust = 0.5),
  list(
    width = 0.55, item = "forest", hjust = 0.5, heading = "Hazard ratio", linetype = "dashed",
    line_x = 0
  ),
  list(width = 0.03, item = "vline", hjust = 0.5),
  list(width = 0.12, display = ~ ifelse(reference, "Reference", sprintf(
    "\%0.2f (\%0.2f, \%0.2f)",
    trans(estimate), trans(conf.low), trans(conf.high)
  )), display_na = NA),
  list(
    width = 0.05,
    display = ~ ifelse(reference, "", format.pval(p.value, digits = 1, eps = 0.001)),
    display_na = NA, hjust = 1, heading = "p"
  ),
  list(width = 0.03)
)
forest_model(coxph(Surv(time, status) ~ ., pretty_lung), panels)

data_for_lm <- tibble(
  x = rnorm(100, 4),
  y = rnorm(100, 3, 0.5),
  z = rnorm(100, 2, 2),
  outcome = 3 * x - 2 * y + 4 * z + rnorm(100, 0, 0.1)
)

print(forest_model(lm(outcome ~ ., data_for_lm)))

data_for_logistic <- data_for_lm \%>\% mutate(
  outcome = (0.5 * (x - 4) * (y - 3) * (z - 2) + rnorm(100, 0, 0.05)) > 0.5
)

print(forest_model(glm(outcome ~ ., binomial(), data_for_logistic)))
}
