% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ff.R
\name{ff}
\alias{ff}
\alias{ff.default}
\title{Fuzzy forests algorithm}
\usage{
\method{ff}{default}(X, y, Z = NULL, module_membership,
  screen_params = screen_control(min_ntree = 500),
  select_params = select_control(min_ntree = 500), final_ntree = 5000,
  num_processors = 1, nodesize, test_features = NULL, test_y = NULL,
  ...)

ff(X, ...)
}
\arguments{
\item{X}{A data.frame.
Each column corresponds to a feature vectors.}

\item{y}{Response vector.  For classification, y should be a
factor.  For regression, y should be
numeric.}

\item{Z}{A data.frame. Additional features that are not to be
screened out at the screening step.}

\item{module_membership}{A character vector giving the module membership of
each feature.}

\item{screen_params}{Parameters for screening step of fuzzy forests.
See \code{\link[fuzzyforest]{screen_control}} for
details. \code{screen_params} is an object of type
\code{screen_control}.}

\item{select_params}{Parameters for selection step of fuzzy forests.
See \code{\link[fuzzyforest]{select_control}} for details.
\code{select_params} is an object of type
\code{select_control}.}

\item{final_ntree}{Number of trees grown in the final random forest.
This random forest contains all selected features.}

\item{num_processors}{Number of processors used to fit random forests.}

\item{nodesize}{Minimum terminal nodesize. 1 if classification.
5 if regression.  If the sample size is very large,
the trees will be grown extremely deep.
This may lead to issues with memory usage and may
lead to significant increases in the time it takes
the algorithm to run.  In this case,
it may be useful to increase \code{nodesize}.}

\item{test_features}{A data.frame containing features from a test set.
The data.frame should contain the features in both
X and Z.}

\item{test_y}{The responses for the test set.}

\item{...}{Additional arguments currently not used.}
}
\value{
An object of type \code{\link[fuzzyforest]{fuzzy_forest}}.  This
object is a list containing useful output of fuzzy forests.
In particular it contains a data.frame with a list of selected the features.
It also includes a random forest fit using the selected features.
}
\description{
Fits the fuzzy forests algorithm. Note that a formula interface for
fuzzy forests also exists: \code{\link[fuzzyforest]{ff.formula}}.
}
\note{
This work was partially funded by NSF IIS 1251151 and AMFAR 8721SC.
}
\examples{
#ff requires that the partition of the covariates be previously determined.
#ff is also handy if the user wants to test out multiple settings of WGCNA
#prior to running fuzzy forests.

library(mvtnorm)
gen_mod <- function(n, p, corr) {
  sigma <- matrix(corr, nrow=p, ncol=p)
  diag(sigma) <- 1
  X <- rmvnorm(n, sigma=sigma)
  return(X)
}

gen_X <- function(n, mod_sizes, corr){
  m <- length(mod_sizes)
  X_list <- vector("list", length = m)
  for(i in 1:m){
    X_list[[i]] <- gen_mod(n, mod_sizes[i], corr[i])
  }
  X <- do.call("cbind", X_list)
  return(X)
}

err_sd <- .5
n <- 500
mod_sizes <- rep(25, 4)
corr <- rep(.8, 4)
X <- gen_X(n, mod_sizes, corr)
beta <- rep(0, 100)
beta[c(1:4, 76:79)] <- 5
y <- X\%*\%beta + rnorm(n, sd=err_sd)
X <- as.data.frame(X)

Xtest <- gen_X(n, mod_sizes, corr)
ytest <- Xtest\%*\%beta + rnorm(n, sd=err_sd)
Xtest <- as.data.frame(Xtest)

cdist <- as.dist(1 - cor(X))
hclust_fit <- hclust(cdist, method="ward.D")
groups <- cutree(hclust_fit, k=4)

screen_c <- screen_control(keep_fraction = .25,
                           ntree_factor = 1,
                           min_ntree = 250)
select_c <- select_control(number_selected = 10,
                           ntree_factor = 1,
                           min_ntree = 250)
\donttest{
ff_fit <- ff(X, y, module_membership = groups,
             screen_params = screen_c,
             select_params = select_c,
             final_ntree = 250)
#extract variable importance rankings
vims <- ff_fit$feature_list

#plot results
modplot(ff_fit)

#obtain predicted values for a new test set
preds <- predict(ff_fit, new_data=Xtest)

#estimate test set error
test_err <- sqrt(sum((ytest - preds)^2)/n)
}
}
\references{
Conn, D., Ngun, T., Ramirez C.M., Li, G. (2019).
"Fuzzy Forests: Extending Random Forest Feature Selection for Correlated, High-Dimensional Data."
\emph{Journal of Statistical Software}, \strong{91}(9).
\doi{doi:10.18637/jss.v091.i09}

Breiman, L. (2001).
"Random Forests."
\emph{Machine Learning}, \strong{45}(1), 5-32.
\doi{doi:10.1023/A:1010933404324}

Zhang, B. and Horvath, S. (2005).
"A General Framework for Weighted Gene Co-Expression Network Analysis."
\emph{Statistical Applications in Genetics and Molecular Biology}, \strong{4}(1).
\doi{doi:10.2202/1544-6115.1128}
}
\seealso{
\code{\link[fuzzyforest]{ff.formula}},
         \code{\link[fuzzyforest]{print.fuzzy_forest}},
         \code{\link[fuzzyforest]{predict.fuzzy_forest}},
         \code{\link[fuzzyforest]{modplot}}
}
