#' @details This function performs a one-sample sign test on ranked set samples. For balanced RSS (BRSS), it uses the method introduced by Hettmansperger (1995), while for unbalanced RSS (URSS), it follows the approach described by Barabesi (2001). Provide `data` as a data frame with columns `rank` and `y`. The function calculates the sign statistic, confidence intervals, and p-value based on the RSS data.
#' @title RSS Sign test
#' @name rss.sign.test
#' @description The rss.sign.test function performs Sign test on ranked set sample data, supporting both balanced and unbalanced RSS designs.
#'
#' @param data A numeric data frame of ranked set samples with columns `rank` for ranks and `y` for data values.
#' @param alpha A numeric value specifying the confidence level for the interval.
#' @param alternative A character string specifying the alternative hypothesis. Must be one of "two.sided" (default), "greater", or "less".
#' @param median0 A numeric value indicating the hypothesized median for the one-sample test.
#'
#' @return
#'   \item{RSS_median}{The RSS median estimate.}
#'   \item{sign}{The sign-statistic for the test.}
#'   \item{CI}{The confidence interval for the population median.}
#'   \item{z}{The z-statistic for the test.}
#'   \item{p.value}{The p-value for the test.}
#'
#' @references
#'
#' T. P. Hettmansperger. (1995) The ranked-set sample sign test. Journal of Non-parametric Statistics, 4:263–270.
#'
#' L. Barabesi. (2001) The unbalanced ranked-set sample sign test. Journal of Non-parametric Statistics, 13(2):279–289.
#'
#' Chen, Z., Bai Z., Sinha B. K. (2003). Ranked Set Sampling: Theory and Application. New York: Springer.
#'
#' S. Ahn, X. Wang, C. Moon, and J. Lim. (2024) New scheme of empirical likelihood method for ranked set sampling: Applications to two one sample problems. Internation Statistical Review.
#'
#' @seealso
#' \code{\link{rss.simulation}}: used for simulating Ranked Set Samples (RSS), which can serve as input.
#'
#' \code{\link{rss.sampling}}: used for sampling Ranked Set Samples (RSS) from a population data set, providing input data.
#'
#' @importFrom methods is
#' @examples
#' ## Unbalanced RSS with a set size 3 and different sample sizes of 12, 9, 6 for each stratum,
#' ## using imperfect ranking from a lognormal distribution with a mean of 0.
#' rss.data=rss.simulation(H=3,nsamp=c(12,9,6),dist="lognorm", rho=0.8,delta=0)
#'
#' ## RSS sign-test
#' rss.sign.test(data=rss.data, alpha=0.05, alternative="two.sided", median0=0)
#'
#' @export
rss.sign.test <- function(data, alpha=0.05, alternative="two.sided", median0)
{
  alternative.set=c("two.sided", "less", "greater")
  if(!alternative %in% alternative.set) stop("Invalid alternative selected. Please choose from 'two.sided', 'less', or 'greater'.")

  if( (alpha > 0) & (alpha < 1)){

    if(!all(c("rank", "y") %in% colnames(data))) {
      stop("The input data must contain 'rank' and 'y' variables.")
    }
    H = length(unique(data$rank))
    nsamp = table(data$rank)
    n=sum(nsamp)

    sort.y = sort(data$y)

    if( n%%2 ==0 ){
      rss.m = (sort.y[n/2]+sort.y[n/2+1])/2
    }else{
      rss.m = sort.y[(n+1)/2]
    }

    signstat=sum(data$y > median0)

    temp=rep(0,H)
    for(h in (1:H)){
      temp[h]=stats::pbeta(0.5,h,H-h+1)
    }

    if( sum(nsamp == n/H) == H){
      teststat = signstat-n/2
      varstat = 0.25*n*(1-(4/H)*sum((temp-1/2)^2))
      teststat = teststat/sqrt(varstat)

    }else{
      teststat = signstat-sum(nsamp*(1-temp))
      varstat=sum(nsamp*temp*(1-temp))
      teststat = teststat/sqrt(varstat)
    }


    if( n%%2 == 0){
      CI.up = sort.y[ceiling( n/2 + 1 + stats::qnorm(1-alpha/2) * sqrt(varstat) )]
      CI.low = sort.y[ceiling( n/2 - stats::qnorm(1-alpha/2) * sqrt(varstat) )]
    }else{
      CI.up = sort.y[ceiling( floor(n/2) + 1 + stats::qnorm(1-alpha/2) * sqrt(varstat) )]
      CI.low = sort.y[ceiling( floor(n/2) + 1 - stats::qnorm(1-alpha/2) * sqrt(varstat) )]
    }

    if(alternative == "two.sided"){
      pval = 2*(1-stats::pnorm(abs(teststat)))
    }else if(alternative == "less"){
      pval = stats::pnorm(teststat)
    }else if(alternative == "greater"){
      pval = 1-stats::pnorm(teststat)
    }

    result = list(RSS_median = rss.m, sign = signstat, CI = c(CI.low, CI.up),  z = teststat, p.value = pval)
    return(result)
  }else stop("alpha is out of bound.", call. = F)
}
