% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmmsel.R
\name{glmmsel}
\alias{glmmsel}
\title{Generalised linear mixed model selection}
\usage{
glmmsel(
  x,
  y,
  cluster,
  family = c("gaussian", "binomial"),
  local.search = FALSE,
  max.nnz = 100,
  nlambda = 100,
  lambda.step = 0.99,
  lambda = NULL,
  alpha = 0.8,
  intercept = TRUE,
  random.intercept = TRUE,
  standardise = TRUE,
  eps = 1e-04,
  max.cd.iter = 10000,
  max.ls.iter = 100,
  max.bls.iter = 30,
  t.init = 1,
  t.scale = 0.5,
  max.pql.iter = 100,
  active.set = TRUE,
  active.set.count = 3,
  sort = TRUE,
  screen = 100,
  warn = TRUE
)
}
\arguments{
\item{x}{a predictor matrix}

\item{y}{a response vector}

\item{cluster}{a vector of length \code{nrow(x)} with the jth element identifying the cluster
that the jth observation belongs to}

\item{family}{the likelihood family to use; 'gaussian' for a continuous response or 'binomial'
for a binary response}

\item{local.search}{a logical indicating whether to perform local search after coordinate
descent; typically leads to higher quality solutions}

\item{max.nnz}{the maximum number of predictors ever allowed to be active}

\item{nlambda}{the number of regularisation parameters to evaluate when \code{lambda} is
computed automatically}

\item{lambda.step}{the step size taken when computing \code{lambda} from the data; should be a
value strictly between 0 and 1; larger values typically lead to a finer grid of subset sizes}

\item{lambda}{an optional vector of regularisation parameters}

\item{alpha}{the hierarchical parameter}

\item{intercept}{a logical indicating whether to include a fixed intercept}

\item{random.intercept}{a logical indicating whether to include a random intercept; applies
only when \code{intercept = TRUE}}

\item{standardise}{a logical indicating whether to scale the data to have unit root mean square;
all parameters are returned on the original scale of the data}

\item{eps}{the convergence tolerance; convergence is declared when the relative maximum
difference in consecutive parameter values is less than \code{eps}}

\item{max.cd.iter}{the maximum number of coordinate descent iterations allowed}

\item{max.ls.iter}{the maximum number of local search iterations allowed}

\item{max.bls.iter}{the maximum number of backtracking line search iterations allowed}

\item{t.init}{the initial value of the gradient step size during backtracking line search}

\item{t.scale}{the scaling parameter of the gradient step size during backtracking line search}

\item{max.pql.iter}{the maximum number of penalised quasi-likelihood iterations allowed}

\item{active.set}{a logical indicating whether to use active set updates; typically lowers the
run time}

\item{active.set.count}{the number of consecutive coordinate descent iterations in which a
subset should appear before running active set updates}

\item{sort}{a logical indicating whether to sort the coordinates before running coordinate
descent; typically leads to higher quality solutions}

\item{screen}{the number of predictors to keep after gradient screening; smaller values typically
lower the run time}

\item{warn}{a logical indicating whether to print a warning if the algorithms fail to
converge}
}
\value{
An object of class \code{glmmsel}; a list with the following components:
\item{beta0}{a vector of fixed intercepts}
\item{gamma0}{a vector of random intercept variances}
\item{beta}{a matrix of fixed slopes}
\item{gamma}{a matrix of random slope variances}
\item{u}{an array of random coefficient predictions}
\item{sigma2}{a vector of residual variances}
\item{loss}{a vector of loss function values}
\item{cd.iter}{a vector indicating the number of coordinate descent iterations for convergence}
\item{ls.iter}{a vector indicating the number of local search iterations for convergence}
\item{pql.iter}{a vector indicating the number of penalised quasi-likelihood iterations for
convergence}
\item{nnz}{a vector of the number of nonzeros}
\item{lambda}{a vector of regularisation parameters used for the fit}
\item{family}{the likelihood family used}
\item{clusters}{a vector of cluster identifiers}
\item{alpha}{the value of the hierarchical parameter used for the fit}
\item{intercept}{whether a fixed intercept is included in the model}
\item{random.intercept}{whether a random intercept is included in the model}
}
\description{
Fits the regularisation path for a sparse generalised linear mixed model (GLMM).
}
\examples{
# Generate data
set.seed(1234)
n <- 100
m <- 4
p <- 10
s <- 5
x <- matrix(rnorm(n * p), n, p)
beta <- c(rep(1, s), rep(0, p - s))
u <- cbind(matrix(rnorm(m * s), m, s), matrix(0, m, p - s))
cluster <- sample(1:m, n, replace = TRUE)
xb <- rowSums(x * sweep(u, 2, beta, '+')[cluster, ])
y <- rnorm(n, xb)

# Fit sparse linear mixed model
fit <- glmmsel(x, y, cluster)
plot(fit)
fixef(fit, lambda = 10)
ranef(fit, lambda = 10)
coef(fit, lambda = 10)
predict(fit, x[1:3, ], cluster[1:3], lambda = 10)
}
\references{
Thompson, R., Wand, M. P., and Wang, J. J. J. (2025). 'Scalable subset selection in
linear mixed models'. arXiv: \href{https://arxiv.org/abs/2506.20425}{2506.20425}.
}
\author{
Ryan Thompson <ryan.thompson-1@uts.edu.au>
}
