% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_hidecan_plot.R
\name{create_hidecan_plot}
\alias{create_hidecan_plot}
\title{Creates a HIDECAN plot}
\usage{
create_hidecan_plot(
  x,
  chrom_length,
  colour_genes_by_score = TRUE,
  remove_empty_chrom = FALSE,
  chroms = NULL,
  chrom_limits = NULL,
  title = NULL,
  subtitle = NULL,
  n_rows = NULL,
  n_cols = 2,
  legend_position = "bottom",
  point_size = 3,
  label_size = 3.5,
  label_padding = 0.15
)
}
\arguments{
\item{x}{A list of \code{GWAS_data_thr}, \code{DE_data_thr} and/or \code{CAN_data_thr}
produced by the \code{\link{apply_threshold}()} function. If named, the names
will be appended to the y-axis labels (use \code{' '} as empty name in the list).}

\item{chrom_length}{Tibble with columns \code{chromosome} and \code{length}, giving
for each chromosome its length in bp (see \code{\link{combine_chrom_length}()}
function).}

\item{colour_genes_by_score}{Logical, whether to colour the genes by score
(\code{TRUE}) or by log2(fold-change) (\code{FALSE}). Default value is \code{TRUE}.}

\item{remove_empty_chrom}{Logical, should chromosomes with no significant
markers/genes nor candidate genes be removed from the plot? Default value
if \code{FALSE}.}

\item{chroms}{Character vector, name of chromosomes to include in the plot.}

\item{chrom_limits}{Integer vector of length 2, or named list where the
elements are integer vectors of length 2. If vector, gives the lower and upper
limit of the chromosomes (in bp) to use in the plot. If a named list, names
should correspond to chromosome names. Gives for each chromosome the lower
and upper limits (in bp) to use in the plot. Doesn't have to be specified
for all chromosomes. Default value is \code{NULL}, i.e. no limits are applied
to the chromosomes (they will be plotted in their entirety).}

\item{title}{Character, title of the plot. Default value is \code{NULL} (i.e.
no title will be added to the plot).}

\item{subtitle}{Character, subtitle of the plot. Default value is \code{NULL}
(i.e. no subtitle will be added to the plot).}

\item{n_rows}{Integer, number of rows of chromosomes to create in the plot.
Default value is \code{NULL}.}

\item{n_cols}{Integer, number of columns of chromosomes to create in the plot.
Default value is 2. Will be set to \code{NULL} if \code{n_rows} is not \code{NULL}.}

\item{legend_position}{Character, position of the legend in the plot. Can be
\code{bottom} (default value), \code{top}, \code{right}, \code{left} or \code{none}.}

\item{point_size}{Numeric, size of the points in the plot. Default value is 3.}

\item{label_size}{Numeric, size of the gene labels in the plot. Default value is
3.5 (for \code{\link[ggrepel]{geom_label_repel}}).}

\item{label_padding}{Numeric, amount of padding around gene labels in the plot,
as unit or number. Default value is 0.15
(for \link[ggrepel]{geom_label_repel}).}
}
\value{
A \code{ggplot}.
}
\description{
Creates a HIDECAN plot from a list of filtered GWAS or DE results
and/or candidate genes.
}
\examples{
if (interactive()) {
x <- get_example_data()
y <- list("GWAS" = GWAS_data(x[["GWAS"]]),
          "DE" = DE_data(x[["DE"]]),
          "CAN" = CAN_data(x[["CAN"]]))

chrom_length <- combine_chrom_length(y)

z <- list(
  apply_threshold(y[["GWAS"]], score_thr = 4),
  apply_threshold(y[["DE"]], score_thr = 1.3, log2fc_thr = 0.5),
  apply_threshold(y[["CAN"]])
)

create_hidecan_plot(z,
                    chrom_length,
                    label_size = 2)

## Colour genes according to their fold-change
create_hidecan_plot(z,
                    chrom_length,
                    colour_genes_by_score = FALSE,
                    label_size = 2)

## Add names to the datasets
create_hidecan_plot(setNames(z, c("Genomics", "RNAseq", "My list")),
                    chrom_length,
                    colour_genes_by_score = FALSE,
                    label_size = 2)

## Add names to some of the datasets only (e.g. not for GWAS results)
create_hidecan_plot(setNames(z, c(" ", "RNAseq", "My list")),
                    chrom_length,
                    colour_genes_by_score = FALSE,
                    label_size = 2)

## Set limits on all chromosomes (to "zoom in" to the 10-20Mb region)
create_hidecan_plot(z,
                    chrom_length,
                    label_size = 2,
                    chrom_limits = c(10e6, 20e6))

## Set limits on some chromosomes only
create_hidecan_plot(z,
                    chrom_length,
                    label_size = 2,
                    chrom_limits = list("ST4.03ch00" = c(10e6, 20e6),
                                        "ST4.03ch02" = c(15e6, 25e6)))
}
}
