% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/total_ibd_dist.R
\name{total_ibd_dist}
\alias{total_ibd_dist}
\title{Compute probability distribution of total IBD}
\usage{
total_ibd_dist(
  pedigree,
  ids = pedtools::leaves(pedigree),
  fraction = FALSE,
  states = "ibd",
  ibd_state = 1L,
  chromosome_length = 267.77,
  convolve = TRUE,
  ...
)
}
\arguments{
\item{pedigree}{Pedigree in \code{\link[pedtools:ped]{pedtools::ped}} form.}

\item{ids}{Ids for which IBD is observed. Default is \code{pedtools::leaves(pedigree)}.}

\item{fraction}{If TRUE, the distribution of the IBD fraction instead of length will be returned. Default is FALSE.}

\item{states}{One of \code{"ibd"} (default), \code{"kappa"}, \code{"identity"} or \code{"detailed"}.}

\item{ibd_state}{Default is 1.}

\item{chromosome_length}{Default is 267.77 cM (an estimate of the length of chromosome 1).}

\item{convolve}{Should the distribution of the sum (across chromosomes) be obtained?}

\item{...}{Additional parameters passed to \code{\link{convolve_total_ibd_dists}} when \code{convolve=TRUE}.}
}
\value{
object of class \code{total_ibd_dist}
}
\description{
The \code{total_ibd_dist} function computes the probability distribution
of the total IBD length (or fraction) over one or more autosomes.
}
\details{
For many pedigree relationships, the probability distribution of the
total IBD length over one chromosome is a mixture of two point masses
(0 and chromosome length) and a continuous density.

If \code{convolve=TRUE} (the default) and \code{chromosome_length} has length
greater than one, the convolution of the distributions will be obtained
by FFT using the \code{\link{convolve_total_ibd_dists}} function. Convolution will
typically produce a rapidly increasing number of point masses
with very small probabilities which are discarded if the
probability falls below a threshold of \code{1e-9}; see \code{\link{convolve_total_ibd_dists}}
for details and finer control.
}
\examples{
## Total IBD and fraction of IBD for a cousin relationship
ped_fc <- pedtools::cousinPed()

# total IBD length for 100 cM
dist_length <- total_ibd_dist(ped_fc, chromosome_length = 100)
dist_length
plot(dist_length)

# fraction IBD for 100 cM
dist_fraction <- total_ibd_dist(ped_fc, chromosome_length = 100,
                                fraction = TRUE)
dist_fraction
plot(dist_fraction)

# distribution of total length across three chromosomes (150, 200, 250 cM)
plot(total_ibd_dist(ped_fc, chromosome_length = c(150, 200, 250)))

# a quick approximation with reasonable accuracy (with just 256 gridpoints)
plot(total_ibd_dist(ped_fc, chromosome_length = c(150, 200, 250),
                    number_of_gridpoints_exponent = 8))

## Difference between IBD distributions between half-sibs, uncle-nephew
## and grandparent-grandchild relationships

# kappa1 is 1/2 for half sibs, uncle-nephew and grandparent-grandchild
# but the distribution of the fraction of a chromosome that is in this
# state differs between the relationships

# define pedigrees and verify kappa1
ped_gp <- pedtools::linearPed(n = 2)
ped_av <- pedtools::avuncularPed()
ped_hs <- pedtools::halfSibPed()

stopifnot(all.equal(1/2,
          d_ibd(1, ped_av),
          d_ibd(1, ped_hs),
          d_ibd(1, ped_gp, ids = c(1,5))))

# Compute IBD distributions
d_av <- total_ibd_dist(ped_av)
d_hs <- total_ibd_dist(ped_hs)
d_gp <- total_ibd_dist(ped_gp, ids = c(1,5))

# the point masses are different
d_av
d_hs
d_gp

# plot the continuous densities
x0 <- seq(0, 267.77, length.out = 200)
df <- data.frame(cM = rep(x0, 3),
                 y = c(d(d_av)(x0), d(d_hs)(x0), d(d_gp)(x0)),
                 Relationship = rep(c("Avuncular", "Half-Sibling",
                                      "Grandparent"), each = length(x0)))

require(ggplot2)
ggplot(df, aes(x = cM, y = y, color = Relationship)) + geom_line()
}
