% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/as_ieegio_transform.R
\name{transform_flirt2ras}
\alias{transform_flirt2ras}
\title{Convert FLIRT transform to world (RAS) coordinates}
\usage{
transform_flirt2ras(transform, source = NULL, reference = NULL)
}
\arguments{
\item{transform}{an \code{ieegio_transforms} object with FSL orientation
(typically from \code{\link{io_read_flirt_transform}}), or a 4x4 matrix}

\item{source}{source (moving) image used in FLIRT registration. Can be:
\itemize{
\item A file path to a NIfTI image
\item An \code{ieegio_volume} object
\item \code{NULL} to skip source-side conversion
}}

\item{reference}{reference (fixed) image used in FLIRT registration. Can be:
\itemize{
\item A file path to a NIfTI image
\item An \code{ieegio_volume} object
\item \code{NULL} to skip reference-side conversion
}}
}
\value{
An \code{ieegio_transforms} object with updated orientations:
\itemize{
\item Both images provided: RAS -> RAS transform
\item Source only: RAS -> FSL transform (source side converted)
\item Reference only: FSL -> RAS transform (reference side converted)
\item Neither: FSL -> FSL transform (unchanged)
}
}
\description{
Converts an FSL FLIRT matrix from FSL scaled-voxel coordinates to world
(RAS) coordinates. Allows partial conversion by specifying only source,
only reference, or both images.
}
\details{
FSL FLIRT matrices operate in a scaled-voxel coordinate system that depends
on the image geometry. The conversion to world coordinates uses:

\code{world_transform = ref_vox2ras \%*\% ref_fsl2vox \%*\% flirt \%*\% src_vox2fsl \%*\% src_ras2vox}

Where:
\itemize{
\item \code{src_ras2vox}: Inverse of source image's voxel-to-RAS matrix
\item \code{src_vox2fsl}: Source voxel-to-FSL coordinate transform
\item \code{flirt}: The original FLIRT matrix
\item \code{ref_fsl2vox}: Inverse of reference voxel-to-FSL transform
\item \code{ref_vox2ras}: Reference image's voxel-to-RAS matrix
}

The FSL coordinate system uses scaled voxels with possible X-axis flip
depending on the image's \code{sform} determinant sign.
}
\examples{
\dontrun{
# Read FLIRT matrix
xfm <- io_read_flirt_transform("source_to_reference.mat")

# Full conversion to RAS coordinates
xfm_ras <- transform_flirt2ras(xfm,
                               source = "source.nii.gz",
                               reference = "reference.nii.gz")

# Partial conversion (reference side only)
xfm_partial <- transform_flirt2ras(xfm, reference = "reference.nii.gz")

# Using ieegio_volume objects
src_vol <- read_volume("source.nii.gz", header_only = TRUE)
ref_vol <- read_volume("reference.nii.gz", header_only = TRUE)
xfm_ras <- transform_flirt2ras(xfm, source = src_vol, reference = ref_vol)
}

}
\seealso{
\code{\link{io_read_flirt_transform}} for reading FLIRT matrices
\code{\link{transform_orientation}} for general orientation transforms
}
