#' Jabot's trait model
#'
#' Function `traitSim` (written in C++) simulates a realization from
#' the trait model suggested by Jabot (2010); function `traiStat`
#' computes a possible summary statistics.
#'
#' The model describes the distribution of a numeric trait in a population
#' of size `population`. The trait can only assume the values
#' \eqn{(i-1)/(nspecies-1)} for \eqn{i=1,...,nspecies}. The local
#' competitive ability of a species with trait `u` is proportional to
#' \deqn{F(u)=1-\omega+\omega\phi(u; \mu, \sigma)} where \eqn{\omega},
#' \eqn{\mu} and \eqn{\sigma} are parameters, and \eqn{\phi(u;\mu,
#' \sigma)} denotes the probability density function of a normal random variable with mean
#' \eqn{\mu} and variance \eqn{\sigma^2}.
#'
#' The traits of the initial
#' population are randomly drawn with probability proportional to
#' \eqn{F(u)}. Then, for `ngen` steps, one individual randomly chosen
#' dies. It is replaced either by an immigrant (with probability
#' \eqn{\gamma}) or by a descendant of another randomly choosen existing
#' individual (with probability \eqn{1-\gamma}).
#' In the first case (immigration), the trait of the new individual is drawn with
#' probability proportional to \eqn{F(u)}. In the second case
#' (reproduction), the the probability that the trait of
#' the new individual is `u` is proportional to the abundance of `u` in the
#' population times \eqn{F(u)}.  
#'
#' The vector of model parameters is \eqn{\theta=(\gamma, \mu, \sigma, \omega)'}.
#' Note that the parametrization used in this package differs from the one
#' originally suggested  by Jabot (2010). Specifically, Jabot assumes that
#' \eqn{\gamma=J/(J+population-1)} and
#' \eqn{F(u)=1+2A\pi\sigma\phi(u;\mu, \sigma)}
#' where \eqn{J} and \eqn{A} are alternative parameters used in place of
#' \eqn{\gamma} and \eqn{\omega}, respectively.
#' @name trait-model
#' @aliases trait
#' @references
#' Franck Jabot (2010) 'A Stochastic Dispersal-Limited Trait-Based Model of
#' Community Dynamics', Journal of Theoretical Biology,  262, pp.
#' 650–61, \doi{10.1016/j.jtbi.2009.11.004}.
#' @keywords model
NULL

#' @rdname trait-model
#' @param n the observed data
#' @param q the quantiles used to summarize the trait distribution
#' @returns
#' `traitStat` returns a numeric vector containing the species richness
#' (i.e., the number of distinct traits in the population),  
#' the Gini index, and the quantiles of the trait
#' characteristic corresponding to the input arguments `q`
#' @examples
#' set.seed(1)
#' theta <- c(0.2, 0.7, 0.2, 0.7)
#' y <- traitSim(theta)
#' plot(seq(0, 1, length = length(y)), y, type = "h", xlab = "trait", ylab = "frequency")
#' print(tobs <- traitStat(y))
#' \donttest{
#' # It takes some time to run
#' tsim <- function(theta) traitStat(traitSim(theta))
#' m <- ifit(tobs, tsim, l = rep(0, 4), u = rep(1, 4), trace = 1000)
#' m
#' confint(m)
#' diagIFIT(m)
#' numsimIFIT(m)
#' }
#' @export
traitStat <- function(n, q = c(0.01, seq(0.05, 0.95, by = 0.05), 0.99)) {
    p <- n[n > 0] / sum(n)
    z <- rep(seq(0, 1, length = length(n)), n)
    c(length(p), 1 - sum(p * p), stats::quantile(z, q))
}
