nnlm <- function(data, nfit, guess, complete = FALSE) {
    p <- data$p
    n <- data$n
    pscale <- pmax(1, abs(guess))
    par <- data$par[, seq.int(n), drop=FALSE] - guess
    idx <- knnsearch0(par / pscale, nfit)
    m <- stats::lm(t(data$stat[, idx, drop=FALSE]) ~ t(par[, idx, drop=FALSE]))
    B <- matrix(stats::coef(m), nrow=p+1)
    B[!is.finite(B)] <- 0
    ans <- list(a = B[1, ], B = t(B[-1, , drop = FALSE]))
    if (complete) {
        ans$scale <- pscale
        ans$V <- crossprod(stats::residuals(m) / sqrt(nfit - p - 1))
        ia <- seq(1, data$q * (p + 1), by = p + 1)
        ans$Va <- stats::vcov(m)[ia, ia]
    }
    ans
}

sqm <- function(A, eps = sqrt(.Machine$double.eps)) {
    A <- eigen(A, symmetric = TRUE)
    n <- sum(A$values > eps)
    d <- c(sqrt(A$values[seq_len(n)]), rep(0, length(A$values) - n))
    A$vectors %*% (d * t(A$vectors))
}

isqm <- function(A, eps = sqrt(.Machine$double.eps)) {
    A <- eigen(A, symmetric = TRUE)
    n <- sum(A$values > eps)
    d <- c(1 / sqrt(A$values[seq_len(n)]), rep(0, length(A$values) - n))
    A$vectors %*% (d * t(A$vectors))
}

## truncate multivariate normal
tnorm <- function(n, mu, S05, l, u) {
    repeat {
        x <- mu + S05 %*% stats::rnorm(n)
        if (all(l < x, x < u)) break
    }
    x
}

fastrob <- function(x) {
    s <- pmax(.Machine$double.eps, apply(x, 1, stats::mad))
    outer(s, s) * stats::cor(stats::qnorm(apply(x, 1, rank) / (NCOL(x) + 1)))
}

checkIFIT <- function(
        p, q, l, u,
        trace, Ntotal, NTotglobal, Ninit, Nelite, Aelite,
        Tolglobal, Tollocal, Tolmodel,
        NFitlocal, NAddglobal, NAddlocal, Rhomax, Lambda
    ) {
    err <- NULL
    if ((p <= 0) || (q <= 0)) {
        err <- c(err, "At least one parameter and one summary statistic must be defined")
    }
    if (p > q) {
        err <- c(err, "the number of summary statistics must be equal or greater than the number of parameters")
    }
    if (length(l) != length(u)) {
        err <- c(err, "the length of the lower and upper bounds must be equal")
    }
    if ((Ntotal <= 0) || (NTotglobal <= 0) || (Ninit <= 0) || (Nelite <= 0) ||
        (Tolglobal <= 0) || (Tollocal <= 0) || (Tolmodel <= 0) ||
        (NFitlocal <= 0) || (NAddglobal <= 0) || (NAddlocal <= 0) || (Rhomax <= 0)) {
        err <- c(err, "Ntotal, NTotglobal, Ninit, Nelite, Tolglobal, Tollocal, Tolmodel, NFitlocal, Naddglobal, Naddlocal, and Rhomax must be greater than 0")
    }
    if (Ntotal <= NTotglobal) {
        err <- c(err, "Ntotal must be greater than Ntotglobal")
    }
    if (Ntotal <= NFitlocal) {
        err <- c(err, "Ntotal must be greater than NFitlocal")
    }
    if ((Lambda <= 0) || (Lambda > 1)) {
        err <- c(err, "Lambda must be in (0, 1]")
    }
    if (length(err)) {
        writeLines(err)
        stop("there are errors in the input of ifit")
    } else {
        list(
            trace = trace, Ntotal = Ntotal, NTotglobal = NTotglobal,
            Ninit = Ninit, Nelite = Nelite, Aelite = Aelite,
            Tolglobal = Tolglobal, Tollocal = Tollocal, Tolmodel = Tolmodel,
            NFitlocal = NFitlocal, NAddglobal = NAddglobal, NAddlocal = NAddlocal,
            Rhomax = Rhomax, Lambda = Lambda
        )
    }
}
