% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R, R/enzyme.R
\name{enzyme-model}
\alias{enzyme-model}
\alias{enzymeSim}
\alias{enzyme}
\alias{enzymeStat}
\title{Michaelis-Menten enzyme kinetics}
\usage{
enzymeSim(theta, n = 50L, E0 = 100L, S0 = 100L)

enzymeStat(y, degree = 2, knots = 0.2)
}
\arguments{
\item{theta}{a vector of length 3 containing the model parameters}

\item{n}{the process is sampled on n equispaced points on the unit interval
(default 50)}

\item{E0, S0}{initial state of the process
(default E0=100, S0=100)}

\item{y}{a \verb{n x 4} matrix containing the observed data}

\item{degree}{the degree of the B-splines basis (default 2)}

\item{knots}{the knots of the B-splines basis (default: only one knot equal to 0.2)}
}
\value{
\code{enzymeSim} returns a `n x 4' integer matrix; the ith row contains the
simulated values for E, S, C and P at time t=(i-1)/(n-1).

\code{enzymeStat} returns a numeric vector of length
2*(degree+1+length(knots)) containing the coefficents
of the B-splines curves (with degree and knots given by the second and third
arguments) fitted by least squares to the complex and product trajectories
(i.e., to the last two components of the process's state).
}
\description{
Function \code{enzymeSim} (written in C++) simulates a realization from
the stochastic Michaelis-Menten kinetics model; function \code{enzymeStat}
computes a possible summary statistics.
}
\details{
The model describes a simple case of enzyme kinetics. It involves an
enzyme E binding to a substrate S to form a complex C. This complex C
then releases a product P while simultaneously regenerating the original
enzyme E. The possible reactions are \code{E + S -> C}, \code{C -> E + S}, and \code{C -> P + E},
with rates that constitute the three model parameters.

In probabilistic terms, the integer-valued vector
\eqn{(E_t, S_t, C_t, P_t)'} is generated by a continuous-time Markov process
such that:
\deqn{
\begin{aligned}
&Pr\{E_{t+\delta}=E_t-1, S_{t+\delta}=S_t-1,
C_{t+\delta}=C_t+1, P_{t+\delta}=P_t
|E_t, S_t, C_t, P_t\} \\&\;\;\;\;\;\;\;\;=
\theta_1 E_tS_t\delta+o(\delta),  \\
&Pr\{E_{t+\delta}=E_t+1, S_{t+\delta}=S_t+1,
C_{t+\delta}=C_t-1, P_{t+\delta}=P_t
|E_t, S_t, C_t, P_t\} \\&\;\;\;\;\;\;\;\;=
\theta_2 C_t\delta+o(\delta),  \\
&Pr\{E_{t+\delta}=E_t+1, S_{t+\delta}=S_t,
C_{t+\delta}=C_t-1, P_{t+\delta}=P_t+1
|E_t, S_t, C_t, P_t\} \\&\;\;\;\;\;\;\;\;=
\theta_3 C_t\delta+o(\delta).
\end{aligned}
}
The initial state is \eqn{(E_0, S_0, C_0, P_0)'=(E0, S0, 0, 0)'}.
Process is simulated using Gillespie exact algorithm.
}
\note{
The summary statistics is based only on \eqn{C_t} and \eqn{P_t}.
Indeed, at every time point \eqn{t},  \eqn{E_t=E0-C_t} and
\eqn{S_t=S0-C_t-P_t}.
}
\examples{
set.seed(1)
theta <- c(0.5, 2.5, 1)
y <- enzymeSim(theta)
plot(ts(y, start = 0, frequency = 50), main = "")
print(tobs <- enzymeStat(y))
\donttest{
# It takes some time to run
tsim <- function(theta) enzymeStat(enzymeSim(theta))
m <- ifit(tobs, tsim, l = rep(0, 3), u = rep(50, 3), trace = 1000)
m
confint(m)
diagIFIT(m)
numsimIFIT(m)
}
}
\references{
Darren J. Wilkinson (2019) Stochastic Modelling for Systems Biology,
Third edition, CRC Press
}
\keyword{model}
