% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutex.r
\name{mutex}
\alias{mutex}
\alias{with.mutex}
\title{Shared and Exclusive Locks}
\usage{
mutex(name = uid(), assert = NULL, cleanup = FALSE, file = NULL)

\method{with}{mutex}(data, expr, alt_expr = NULL, shared = FALSE, timeout_ms = Inf, ...)
}
\arguments{
\item{name}{Unique ID. Alphanumeric, starting with a letter.}

\item{assert}{Apply an additional constraint.
\itemize{
\item \code{'create'} - Error if the mutex \emph{already exists}.
\item \code{'exists'} - Error if the mutex \emph{doesn't exist}.
\item \code{NULL} - No constraint; create the mutex if it doesn't exist.
}}

\item{cleanup}{Remove the mutex when the R session exits. If \code{FALSE},
the mutex will persist until \verb{$remove()} is called or the operating
system is restarted.}

\item{file}{Use a hash of this file/directory path as the mutex name. The
file itself will not be read or modified, and does not need to exist.}

\item{data}{A \code{mutex} object.}

\item{expr}{Expression to evaluate if the mutex is acquired.}

\item{alt_expr}{Expression to evaluate if \code{timeout_ms} is reached.}

\item{shared}{If \code{FALSE} (the default) an exclusive lock is returned.
If \code{TRUE}, a shared lock is returned instead. See description.}

\item{timeout_ms}{Maximum time (in milliseconds) to block the process
while waiting for the operation to succeed. Use \code{0} or \code{Inf} to
return immediately or only when successful, respectively.}

\item{...}{Not used.}
}
\value{
\code{mutex()} returns a \code{mutex} object with the following methods:
\itemize{
\item \verb{$name}
\itemize{
\item Returns the mutex's name (scalar character).
}
\item \verb{$lock(shared = FALSE, timeout_ms = Inf)}
\itemize{
\item Returns \code{TRUE} if the lock is acquired, or \code{FALSE} if the timeout is reached.
}
\item \verb{$unlock(warn = TRUE)}
\itemize{
\item Returns \code{TRUE} if successful, or \code{FALSE} (with optional warning) if the mutex wasn't locked by this process.
}
\item \verb{$remove()}
\itemize{
\item Returns \code{TRUE} if the mutex was successfully deleted from the operating system, or \code{FALSE} on error.\cr\cr
}
}

\code{with()} returns \code{eval(expr)} if the lock was acquired, or \code{eval(alt_expr)} if the timeout is reached.
}
\description{
Mutually exclusive (mutex) locks are used to control access to shared
resources.\cr\cr
An \emph{exclusive lock} grants permission to one process at a time, for
example to update the contents of a database file. While an exclusive lock
is active, no other exclusive or shared locks will be granted.\cr\cr
Multiple \emph{shared locks} can be held by different processes at the same
time, for example to read a database file. While a shared lock is active, no
exclusive locks will be granted.
}
\details{
The operating system ensures that mutex locks are released when a process
exits.
}
\section{Error Handling}{


The \code{with()} wrapper automatically unlocks the mutex if an error stops
evaluation of \code{expr}. If you are directly calling \code{lock()}, be sure that
\code{unlock()} is registered with error handlers or added to \code{on.exit()}.
Otherwise, the lock will persist until the process terminates.
}

\section{Duplicate Mutexes}{


Mutex locks are per-process. If a process already has a lock, it can not
attempt to acquire a second lock on the same mutex.
}

\examples{

tmp <- tempfile()
mut <- interprocess::mutex(file = tmp)

print(mut)

# Exclusive lock to write the file
with(mut, writeLines('some data', tmp))

# Use a shared lock to read the file
with(mut,
  shared     = TRUE,
  timeout_ms = 0, 
  expr       = readLines(tmp), 
  alt_expr   = warning('Mutex was locked. Giving up.') )

# Directly lock/unlock with safeguards
if (mut$lock(timeout_ms = 0)) {
  local({
    on.exit(mut$unlock())
    writeLines('more data', tmp)
  })
} else {
  warning('Mutex was locked. Giving up.')
}

mut$remove()
unlink(tmp)
}
\seealso{
Other shared objects: 
\code{\link{msg_queue}()},
\code{\link{semaphore}()}
}
\concept{shared objects}
