% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/intKrigeDriver.R
\name{intkrige}
\alias{intkrige}
\title{Algorithmic implementation of interval valued kriging.}
\usage{
intkrige(
  locations,
  newdata,
  models,
  centerFormula = center ~ 1,
  eta = 0.75,
  A = c(1, 1, 0),
  trend = NULL,
  thresh = 100,
  tolq = 0.001,
  maxq = 100,
  tolp = 0.001,
  maxp = 100,
  r = 1,
  useR = TRUE,
  fast = FALSE,
  weights = FALSE,
  cores = 1,
  progress = FALSE
)
}
\arguments{
\item{locations}{An object of class intsp, specifying the prediction
locations with an interval-valued response.}

\item{newdata}{An object of class SpatialPointsDataFrame or
SpatialPixelsDataFrame specifying the locations at which to predict
intervals.}

\item{models}{A list of variogram models of class vgm (see \link[gstat]{vgm})
When specified, the third model represents the center/radius interaction.}

\item{centerFormula}{The formula describing the relationship between the radius center
and any number of explanatory variables. Must have "center" as the lone
dependent variable. Note that there is no option to fit an external
trend to the radius.}

\item{eta}{A growth/shrink parameter for penalty term.
For simple kriging: eta > 1. For ordinary kriging: eta < 1.}

\item{A}{vector of length three representing the weights
of the generalized L2 distance: the vector of three contains the weights for
the center, radius, and center/radius respectively.
A = c(1, 1, 0) assumes the regular L2 distance calculation for intervals.}

\item{trend}{If null, use ordinary kriging. When specified, represents the
known mean of the stationary process and is an indication to use
simple kriging.}

\item{thresh}{Let n = length(locations). When abs(lam_i) < 1/(n*thresh),
this lambda value is set to 0.}

\item{tolq}{For a set penalty term, convergence is satisfied if
max(abs(lamUp-lam)) < tolq.}

\item{maxq}{For a set penalty term, the max number of iterations
allowed for convergence.}

\item{tolp}{When abs(sum(abs(lam)) - 1) < tolp, consider the
constraints satisfied.}

\item{maxp}{Maximum number of allowed iterations to satisfy
equation constraints.}

\item{r}{The starting value of the penalty term. Must be relatively large to
ensure that the initial solution stays within the feasible region.}

\item{useR}{If TRUE, use the R version of the algorithm.
If FALSE, use the rcppArmadillo version.}

\item{fast}{(Simple kriging only). If TRUE, allows lambdas to converge to 0
and subsets matrices accordingly. When FALSE, runs simple kriging using a
barrier penalty at 0. Fast = TRUE is orders of magnitude faster than the
full implementation. However, it is not recommended when input
measurements are sparse as it is known to have convergence issues
in these cases.}

\item{weights}{If TRUE, return the vector kriging weights for each prediction.
If false, simply return the predicted output.}

\item{cores}{An integer (for parallel computing): specify the number
of cores that will be devoted to the computation.
Note that the argument 'all' will
use all available cores minus one.
Parallel processing is only relevant if you are predicting
for more than one location.
Note there is no parallel option when useR = FALSE.}

\item{progress}{(Not valid when useR=FALSE or in parallel):
Print a progress bar showing the progress of the predictions
at the new locations.}
}
\value{
A matrix with 4 columns where rows correspond to the prediction
 locations and columns correspond to:

- center prediction

- radius predictions

- kriging prediction variance

- warning column for non-convergent optimization problem
(0 - no warning, 1 - warning)
}
\description{
Function to implement the interval valued extension of ordinary and
simple kriging. Includes all necessary input checks and error handling.
Essentially acts as a switch function between the R and c++ versions
of the algorithm.
}
\details{
The centerFormula argument allows for an external trend to be fit to the
  interval centers. No option is provided to scale the interval radii. This
  means that any transformations should be applied to the entire interval
  prior to input into interval-valued kriging. This ensures that input into
  the interval-valued kriging algorithm are well-defined intervals with
  properly ordered upper and lower endpoints.
}
\examples{
# First, define the location and elevation of interest.
# (In this case we pick coordinates of Utah State University)
tproj <- sp::CRS(SRS_string = "EPSG:4326")
templocs <- data.frame(lat = 41.745, long = -111.810, ELEVATION = 1456)
sp::coordinates(templocs) <- c("long", "lat")
sp::proj4string(templocs) <- tproj

# Load the Utah Snow Load Data
data(utsnow)
utsnow.sp <- utsnow

# Convert to an 'intsp' object that inherits a SpatialPointsDataFrame
sp::coordinates(utsnow.sp) <- c("LONGITUDE", "LATITUDE")
sp::proj4string(utsnow.sp) <- tproj
interval(utsnow.sp) <- c("minDL", "maxDL")
# Define the formulas we will use to define the intervals.
temp_formula <- center ~ ELEVATION

# Define, fit and check the variogram fits.
varios <- intvariogram(utsnow.sp,
                       centerFormula = temp_formula)
varioFit <- fit.intvariogram(varios, models = gstat::vgm(c("Sph", "Sph", "Gau")))
preds <- intkrige::intkrige(locations = utsnow.sp,
newdata = templocs,
models = varioFit,
centerFormula = temp_formula)

}
