% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/find_best_reconstruction_QP.R
\name{find_best_reconstruction_QP}
\alias{find_best_reconstruction_QP}
\title{Find "best" reconstruction of a target signature or spectrum from a set of signatures.}
\usage{
find_best_reconstruction_QP(
  target.sig,
  sig.universe,
  max.subset.size = NULL,
  method = "cosine",
  trim.less.than = 1e-10
)
}
\arguments{
\item{target.sig}{The signature or spectrum to reconstruct;
a non-negative numeric vector or 1-column matrix-like object.}

\item{sig.universe}{The universe of signatures from which to reconstruct
\code{target.sig}; a non-negative numeric matrix-like object with
\code{nrow(sig.universe) == length(target.sig)}. The sums of each column
must be 1. Must not contain duplicate columns or have other
non-unique quadratic programming solutions (not checked, but will generate
an error from \code{\link[quadprog]{solve.QP}} in package \code{quadprog}).}

\item{max.subset.size}{Maximum number of signatures to use to
reconstruct \code{target.sig}.}

\item{method}{As in \code{\link[philentropy]{dist_one_one}} in
package \code{philentropy}, and used only
to find the final "best" reconstruction. The optimized exposures from which
to selected the "best" reconstruction are calculated using
\code{\link{optimize_exposure_QP}}, which uses \code{\link[quadprog]{solve.QP}}
in package \code{quadprog}.}

\item{trim.less.than}{After optimizing exposures with
\code{\link{optimize_exposure_QP}}, remove exposures less than
\code{trim.less.than} and then re-optimize.}
}
\value{
A list with elements:
\itemize{
\item \code{optimized.exposure} A numerical vector of the exposures that
give the "best" reconstruction. This vector is empty if there is
an error.
\item \code{similarity} The similarity between the \code{reconstruction}
(see below) and \code{target.sig} according to the distance
or similarity provided by the \code{method} argument.
\item \code{method} The value specified for the \code{method} argument,
or an error message if \code{optimize.exposure} is empty.
\item \code{reconstruction} The reconstruction of \code{target.sig} according to
\code{optimized.exposure}.
}
}
\description{
Find "best" reconstruction of a target signature or spectrum from a set of signatures.
}
\details{
This function should be fast if you do not specify \code{max.subset.size},
but it will be combinatorially slow if \code{max.subset.size} is large
and \code{trim.less.than} is small or negative. So do not do that.
If \code{max.subset.size} is \code{NULL}, then the function just uses \code{\link{optimize_exposure_QP}}.
and then excludes exposures < \code{trim.less.than}, and then re-runs
\code{\link{optimize_exposure_QP}}. Otherwise, after excluding
exposures < \code{trim.less.than}, then the function runs \code{\link{optimize_exposure_QP}} on
subsets of signatures of size <= \code{max.subset.size}, removes exposures < \code{trim.less.than},
reruns \code{\link{optimize_exposure_QP}}, calculates the reconstruction and
similarity between the reconstruction and the \code{target.sig} and returns the information for
the exposures that have the greatest similarity.
}
\examples{
set.seed(888)
sig.u <-
  do.call(
    cbind,
    lapply(1:6, function(x) {
      col <- runif(n = 96)
      col / sum(col)
    })
  )
rr <- find_best_reconstruction_QP(
  target.sig = sig.u[, 1, drop = FALSE],
  sig.universe = sig.u[, 2:6]
)
names(rr)
rr$optimized.exposure
rr$similarity
rr <- find_best_reconstruction_QP(
  target.sig = sig.u[, 1, drop = FALSE],
  sig.universe = sig.u[, 2:6],
  max.subset.size = 3
)
rr$optimized.exposure
rr$similarity

}
