#' Mark correlation functions for homogeneous point patterns with function-valued marks.
#'
#' Mark correlation functions for homogeneous point patterns with function-valued marks.
#'
#' @usage fmcorr(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart",
#'  "isham", "stoyancov", "schlather"),
#' r = NULL,
#' method = c("density","loess"),
#' normalise = TRUE,
#' f = NULL,
#' tol = 0.01,
#' ...)
#' @param X An object of class ppp or lpp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \eqn{r} at which the mark correlation function should be evaluated.
#' @param method Type of smoothing, either \code{density} or \code{loess}. See details.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param tol Tolerance used in the calculation of the conditional mean of the marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' @details
#'
#' The object \code{X} should be an object of class ppp or lpp, with a function-valued mark. The mark should be a \code{data.frame} whose columns give the mark values per time point in an increasing order. The marks per each time point should be numeric.
#' 
#' See detailed information in the help pages of the functions \code{\link[markstat]{mcorr.ppp}} and  \code{\link[markstat]{mcorr.lpp}}. As here, \eqn{X} has a function-valued mark, depending on the class of \eqn{X} being \code{\link[spatstat.geom]{ppp}} or \code{\link[spatstat.linnet]{lpp}}, the function \code{\link[markstat]{mcorr.ppp}} or  \code{\link[markstat]{mcorr.lpp}} will be applied to each time point of the function-valued mark, and at the end an overall mark correlation function will be given. See details in Eckardt et. al (2024).
#'
#' @examples
#'  library(spatstat.random)
#'  library(spatstat.geom)
#'  X <- rpoispp(100)
#'  marks(X) <- data.frame(
#'  t1 = runif(npoints(X),1,10),
#'  t2 = runif(npoints(X),1,10),
#'  t3 = runif(npoints(X),1,10),
#'  t4 = runif(npoints(X),1,10),
#'  t5 = runif(npoints(X),1,10))
#'  fmcorr(X,  ftype = "stoyan", method = "density")

#' @return a data.frame which gives the estimated mark correlation function and the distance vector \eqn{r} at which the mark correlation function is estimated. The outputs of the mark correlation functions for each time point are stored as an attribute, which can be extracted as \code{attr(., "ests")}.
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt
#' @references 
#' Eckardt, M., Mateu, J., & Moradi, M. (2024). Function‐Valued Marked Spatial Point Processes on Linear Networks: Application to Urban Cycling Profiles. Stat, 13(4), e70013.
#' @seealso \code{\link[markstat]{mcorr.ppp}}, \code{\link[markstat]{mcorr.lpp}}, \code{\link[markstat]{fmcorrinhom}}.

#' @import spatstat.univar
#' @import spatstat.linnet
#' @import spatstat.geom
#' @import spatstat.explore
#' @import spatstat.utils
#' @import stats
#' @export

fmcorr <- function(X,
                   ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                   r = NULL,
                   method = c("density","loess"),
                   normalise = TRUE,
                   f = NULL,
                   tol = 0.01,
                   ...){

  if (all(class(X) != "lpp" & class(X) != "ppp")) stop("object X should be of class lpp or ppp.")

  if(all(class(marks(X)) != "data.frame" & class(marks(X)) != "hyperframe")) stop("object X should have a funtion-valued mark as a data.frame whose columns represent marks at time points.")

  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")

  if (missing(method)) stop("smoothing method should be chosen.")

  n <- npoints(X)
  d <- pairdist(X)

  if(is.null(r)){

    if(any(class(X)=="ppp")){

      W <- X$window
      rmaxdefault <- rmax.rule("K", W, n/area(W))
      if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
      breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
      r <- breaks$r

    }else if(any(class(X)=="lpp")){

      L <- X$domain
      rmaxdefault <- 0.98 * boundingradius(L)
      if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
      W <- Window(L)
      breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
      r <- breaks$r

    }else {
      stop("object X should be of class lpp or ppp.")
    }
  }

  rmax <- max(r)

  m <- as.data.frame(marks(X))

  nf <- dim(m)[1]
  f.len <- dim(m)[2]


  if(any(class(X)=="ppp")){

    out <- list()
    for (i in 1:ncol(m)) {

      marks(X) <- as.numeric(m[,i])
      out[[i]] <- mcorr.ppp(X, normalise = normalise, r = r, f = f, ftype = ftype, method = method, tol = tol, ...)
    }

    r <- out[[1]]$r
    emps <- sapply(out, function(df) df$est)
    colnames(emps) <- colnames(m)
    out <- data.frame(r = r, emps)

  }else if(any(class(X)=="lpp")){

    out <- list()
    for (i in 1:ncol(m)) {

      marks(X) <- as.numeric(m[,i])
      out[[i]] <- mcorr.lpp(X, normalise = normalise, r = r, f = f, ftype = ftype, method = method, tol = tol, ...)
    }

    r <- out[[1]]$r
    emps <- sapply(out, function(df) df$est)
    colnames(emps) <- colnames(m)
    out <- data.frame(r = r, emps)

  }else {
    stop("object X should be of class lpp or ppp.")
  }

  finalout <- data.frame(r = r, est = apply(emps, 1, mean))

  
  if(ncol(finalout) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(finalout) <- "mc"
  attr(finalout, "mtype") <- "function-valued"
  attr(finalout, "type") <- type
  attr(finalout, "ests") <- out
  attr(finalout , "ftype") <- ftype
  attr(finalout , "method") <- method
  attr(finalout , "normalise") <- normalise
  
  
  return(finalout)
}
