% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hubert_t.R
\name{matrixStrucTest}
\alias{matrixStrucTest}
\title{Permutation p-values for Gamma and t-statistics}
\usage{
matrixStrucTest(A, group_list = NULL, groups = NULL, B = 1000,
  absolute = TRUE)
}
\arguments{
\item{A}{Distance or similarity matrix, e.g. correlation}

\item{group_list}{List of column indices of A for each group. Either \code{groups} or \code{group_list} but not both must be supplied.}

\item{groups}{CFA model in lavaan syntax. Either \code{groups} or \code{group_list} but not both must be supplied.}

\item{B}{Number of Monte Carlo resamples (defaults to B=1000)}

\item{absolute}{Use the absolute values of A (defaults to TRUE)}
}
\value{
pt_overall_one_sided: Overall one-sided p-value using t statistic

pt_overall_two_sided: Overall two-sided p-value using t statistic

pt_multi_one_sided: Block-specific one-sided p-values using t statistic

pt_multi_two_sided: Block-specific two-sided p-values using t statistic

t0 Observed overall: t statistic

t0k: Observed block-specific t statistic

t_overall: Vector of overall t statistics from permuted A

t_max_one_sided: Vector of max t statistics from permuted A (one-sided)

t_max_two_sided: Vector of max t statistics from permuted A (two-sided)

pG_overall_one_sided: Overall one-sided p-value using Hubert's Gamma

pG_overall_two_sided: Overall two-sided p-value using Hubert's Gamma

pG_multi_one_sided: Block-specific one-sided p-values using Hubert's Gamma

pG_multi_two_sided: Block-specific two-sided p-values using Hubert's Gamma

Gamma0: Observed overall Hubert's Gamma

Gamma0k: Observed block-specific Hubert's Gamma

Gamma_overall: Vector of Hubert's Gamma statistics from permuted A

Gamma_max_one_sided: Vector of max Hubert's Gamma statistics from permuted A (one-sided)

Gamma_max_two_sided: Vector of max Hubert's Gamma statistics from permuted A (two-sided)

B: number of Monte Carlo resamples

group_list: List of column/row indices corresponding to each group
}
\description{
This function computes permutation p-values for Hubert's Gamma and t-statistics
for both overall and block-specific tests.
}
\examples{
# example for matrixStrucTest package
library(matrixStrucTest)
data("big5")

# get column numbers for questionnaire items
items <- grep("[0-9]", colnames(big5))

# compute Spearman's correlation matrix
A <- cor(big5[, items], use = "complete.obs", method = "spearman")

# specify the groups
groups <- "extrovert ~ E1 + E2 + E3 + E4 + E5 + E6 + E7 + E8 + E9 + E10
           neurotic ~ N1 + N2 + N3 + N4 + N5 + N6 + N7 + N8 + N9 + N10
           agreeable ~ A1 + A2 + A3 + A4 + A5 + A6 + A7 + A8 + A9 + A10
           conscientious ~ C1 + C2 + C3 + C4 + C5 + C6 + C7 + C8 + C9 + C10
           open ~ O1 + O2 + O3 + O4 + O5 + O6 + O7 + O8 + O9 + O10"

# compute permutation p-values
# Note: Using small B for fast checking on CRAN. Set B >= 1000 in practice.
result <- matrixStrucTest(A = A, groups = groups, B = 100, absolute = TRUE)

# Note: two-sided p-values from Hubert's Gamma printed by default
#       other results available by directing accessing them from the
#       returned object
result

# Alternative approach for specifying the groups as a list of column/row indices
extrovert <- grep("E", colnames(A))
neurotic <- grep("N", colnames(A))
agreeable <- grep("A", colnames(A))
conscientious <- grep("C", colnames(A))
open <- grep("O", colnames(A))

# put blocks/groups in list
group_list <- list(extrovert = extrovert, 
                   neurotic = neurotic, 
                   agreeable = agreeable, 
                   conscientious = conscientious,
                   open = open)

# Note: Using small B for fast checking on CRAN. Set B >= 1000 in practice.
result <- matrixStrucTest(A = A, group_list = group_list, B = 100, absolute = TRUE)

# Note: two-sided p-values from Hubert's Gamma printed by default
#       other results available by directing accessing them from the
#       returned object
result

# Visualize groups
library(ggplot2)
library(reshape2)

ord <- unlist(result$group_list)
diag(A) <- NA # remove diagonals from color scale
Am <- melt(A[ord, ord])
names(Am) <- c("x", "y", "value")
Am$y <- factor(Am$y, levels = rev(levels(Am$y)))

ggplot(aes(x = x, y = y, fill = abs(value)), data = Am)+
  geom_tile()+
  theme_bw(18)+
  scale_fill_gradient2(space="Lab", name="abs(Cor)", lim = c(0, 1))+
  labs(x = "", y = "")+
  theme(axis.text.x = element_text(angle = 90, vjust = .35,hjust=1))
}
