% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est.sd.R
\name{est.sd}
\alias{est.sd}
\title{Estimating Sample Standard Deviation using Quantiles}
\usage{
est.sd(
   min = NULL, 
   q1 = NULL, 
   med = NULL, 
   q3 = NULL, 
   max = NULL, 
   n = NULL, 
   method = "shi/wan", 
   opt = TRUE
   )
}
\arguments{
\item{min}{numeric value representing the sample minimum.}

\item{q1}{numeric value representing the first quartile of the sample.}

\item{med}{numeric value representing the median of the sample.}

\item{q3}{numeric value representing the third quartile of the sample.}

\item{max}{numeric value representing the sample maximum.}

\item{n}{numeric value specifying the sample size.}

\item{method}{character string specifying the approach used to estimate the sample standard deviations. The options are the following:
\describe{
\item{\code{'shi/wan'}}{The default option. Method of Shi et al. (2020).}
\item{\code{'gld/sld'}}{The method proposed by De Livera et al. (2024). Estimation using the generalised lambda distribution (GLD) for 5-number summaries (\eqn{S_3}), and the skew logistic distribution (SLD) for 3-number summaries (\eqn{S_1} and \eqn{S_2}).}
\item{\code{'wan'}}{The method proposed by Wan et al. (2014).}
\item{\code{'bc'}}{Box-Cox method proposed by McGrath et al. (2020).}
\item{\code{'qe'}}{Quantile Matching Estimation method proposed by McGrath et al. (2020).}
}}

\item{opt}{logical value indicating whether to apply the optimisation step of \code{'gld/sld'} method, in estimating their parameters using theoretical quantiles.
The default value is \code{TRUE}.}
}
\value{
\code{sd}: numeric value representing the estimated standard deviation of the sample.
}
\description{
This function estimates the sample standard deviation from a study presenting quantile summary measures with the sample size (\eqn{n}). The quantile summaries can fall into one of the following categories:
\itemize{
\item \eqn{S_1}: \{ minimum, median, maximum \}
\item \eqn{S_2}: \{ first quartile, median, third quartile \}
\item \eqn{S_3}: \{ minimum, first quartile, median, third quartile, maximum \}
}

The \code{est.sd} function implements newly proposed flexible quantile-based distribution methods for estimating sample standard deviation by De Livera et al. (2024)
as well as other existing methods for estimating sample standard deviations by Shi et al. (2020) and McGrath et al. (2020).
}
\details{
For details explaining the new method \code{'gld/sld'}, check \code{\link{est.mean}}.
}
\examples{
#Generate 5-point summary data
set.seed(123)
n <- 1000
x <- stats::rlnorm(n, 5, 0.5)
quants <- c(min(x), stats::quantile(x, probs = c(0.25, 0.5, 0.75)), max(x))
obs_sd <- sd(x)

#Estimate sample SD using s3 (5 number summary)
est_sd_s3 <- est.sd(min = quants[1], q1 = quants[2], med = quants[3], q3 = quants[4], 
                    max = quants[5], n=n, method = "gld/sld")
est_sd_s3

#Estimate sample SD using s1 (min, median, max)
est_sd_s1 <- est.sd(min = quants[1], med = quants[3], max = quants[5],
                    n=n, method = "gld/sld")
est_sd_s1

#Estimate sample SD using s2 (q1, median, q3)
est_sd_s2 <- est.sd(q1 = quants[2], med = quants[3], q3 = quants[4],
                    n=n, method = "gld/sld")
est_sd_s2


}
\references{
De Livera, A. M., Prendergast, L., & Kumaranathunga, U. (2024). A novel density-based approach for estimating unknown means, distribution visualisations and meta-analyses of quantiles. \emph{arXiv preprint arXiv:2411.10971}. \url{https://arxiv.org/abs/2411.10971}.

Shi, J., Luo, D., Weng, H., Zeng, X.-T., Lin, L., Chu, H., & Tong, T. (2020). Optimally estimating the sample standard deviation from the five-number summary. \emph{Research Synthesis Methods, 11}(5), 641–654.

Wan, X., Wang, W., Liu, J., & Tong, T. (2014). Estimating the sample mean and standard deviation from the sample size, median, range and/or interquartile range. \emph{BMC Medical Research Methodology, 14}, 1–13.

McGrath, S., Zhao, X., Steele, R., Thombs, B. D., Benedetti, A., & the DEPRESSD Collaboration. (2020b). Estimating the sample mean and standard deviation from commonly reported quantiles in meta-analysis. \emph{Statistical Methods in Medical Research, 29}(9), 2520–2537.
}
