#' @name ModelData
#' 
#' @title Create a list giving data to use in Bayesian inference.
#' 
#' @description
#' Function to build the data list to give to stan
#' 
#' @param data An object of class \code{SurvData}
#' @param model_type TKTD GUTS model type ('SD' or 'IT')
#' @param hb_value default is NULL, can be fixed by specifying a numeric.
#' @param \dots Further arguments to be passed to generic methods
#' 
#' @return A list for parameterization of priors for Bayesian inference.
#' 
#' @export
#' 
modelData <- function(data, model_type, ...){
    UseMethod("modelData")
}

#' @name ModelData
#' @export
modelData.SurvData <- function(data, model_type = c("SD", "IT"), hb_value = NULL, ...){
    model_type = match.arg(model_type)
    stanData <- build_stanData(x = data)
    priors <- build_priors(x = data, model_type = model_type, hb_value = hb_value)
    stanData <- c(stanData, priors)
    return(stanData)
}


#' @name ModelData
#' 
#' @title build the data object to be pass for stan
#' 
#' @description
#' 1. Order the data set in replicate and then in time to create a new column
#' \code{i_row} used to delimited replicates.
#' 
#' 2. Create a matrix of replicate and index "id_row"
#' 
#' 3. Compute Nprec = lag of Nsurv
#' 
#' 4. return a list of element to be passed to Stan sampler
#' 
build_stanData <- function(x){
    # order table in replicate > time to create "i_row"
    d <- x[order(x[["replicate"]], x[["time"]]), ]
    n_group = length(unique(d[["replicate"]]))
    subdata <- split(d, d[["replicate"]], drop = TRUE)
    dX <- build_dX(subdata)
    GROUP_X <- group_array(dX)
    dN <- build_dN(subdata)
    GROUP_N <- group_array(dN)

    ls_out <- list(
      n_group = n_group,
      conc = dX$conc,
      time_X = dX$time,
      replicate_X = dX$replicate,
      GROUP_X = GROUP_X,
      n_X = nrow(dX),
      Nsurv = dN$Nsurv,
      Nprec = dN$Nprec,
      time_N = dN$time,
      replicate_N = dN$replicate,
      n_N = nrow(dN),
      GROUP_N = GROUP_N
    )
    return(ls_out)
}


#' @title build sub-dataframe of concentration without NA
#' 
#' @description
#' internal function to
#' remove NA in 'replicate', 'time' and 'conc' columns, remove 'Nsurv' column
#' to only have concentration matrix
#' 
#' @param subdata a list of data.frame
build_dX <- function(subdata){
  lsX <- lapply(seq_along(subdata), function(i){
    d <- stats::na.omit(subdata[[i]][, c("replicate", "time", "conc")])
    return(d)
  })
  dX = do.call("rbind", lsX)
  return(dX)
}

#' @title build sub-dataframe of Nsurv without NA
#' 
#' @description
#' internal function to
#' remove NA in 'replicate', 'time' and 'Nsurv' columns and building 'Nprec'
#' variable. Removing 'conc" column.
#' 
#' @param subdata a list of data.frame
build_dN <- function(subdata){
  lsN <- lapply(seq_along(subdata), function(i){
    d <- stats::na.omit(subdata[[i]][, c("replicate", "time", "Nsurv")])
    d$Nprec <- c(d$Nsurv[1], d$Nsurv[-nrow(d)])
    return(d)
  })
  dN = do.call("rbind", lsN)
  return(dN)
}

#' @title build array with indices of rows
#' 
#' @description
#' internal function to build an array from a data-frame to have the indices 
#' of rows
#' 
#' @param d a data.frame
group_array <- function(d){
  d[["i_row"]] = 1:nrow(d)
  # CREER UN OBJET QUI POUR CHAQUE "replicate" RETOURNE le i_row min et max
  lsGROUP = sapply(unique(d[["replicate"]]), function(u){
    s = d[d[["replicate"]] == u, ]
    out = list(
      i_a = min(s[["i_row"]]),
      i_z = max(s[["i_row"]]),
      n = nrow(s))
    return(out)
  })
  colnames(lsGROUP) = unique(d[["replicate"]])
  # require to pass GROUP as array of int in stan
  # /!\ unlist is require !!
  GROUP = array(unlist(lsGROUP), dim = dim(lsGROUP))
  return(GROUP)
}

#------------------------------------------------------------------------------
#
#                                      PRIORS
#
#------------------------------------------------------------------------------
#' @name ModelData
#' 
#' @title List of priors
#' 
#' @description
#' Create a list of scalars giving priors to use in Bayesian inference.
#'
#' @param x An object of class \code{survData}
#' @param model_type TKTD model type ('SD' or 'IT')
#' 
#' @return A list for parameterization of priors for Bayesian inference with JAGS.
#'
#' @export
build_priors <- function(x, model_type = c("SD", "IT"), hb_value = NULL){
  
  model_type <- match.arg(model_type)
  data <- x[x$time != 0, ]
  # Parameter calculation of concentration min and max
  conc_min <- min(data$conc[data$conc != 0], na.rm = TRUE)
  conc_max <- max(data$conc, na.rm = TRUE)
  time_min <- min(data$time)
  time_max <- max(data$time)
  conc_unic <- sort(unique(data$conc))
  conc_unicPrec <- c(NA, conc_unic[-length(conc_unic)])
  conc_minDelta <- min(conc_unic - conc_unicPrec, na.rm = TRUE)
  
  ## dominant rate constant: kd
  kd_max <- -log(0.001) / time_min
  kd_min <- -log(0.999) / time_max
  ## background hazard rate
  hb_max <- -log(0.5) / time_min
  hb_min <- -log(0.999) / time_max
  ## killing rate parameter: kk
  kk_max <- -log(0.001) / (time_min * conc_minDelta)
  kk_min <- -log(0.999) / (time_max * (conc_max - conc_min))
  ## beta
  beta_minlog10 <- -2
  beta_maxlog10 <- 2

  ## Construction of the list of priors
  priors <-  list(
    ## dominant rate constant: kd
    kd_meanlog10 = priorMean(kd_min, kd_max),
    kd_sdlog10 = priorSD(kd_min, kd_max),
    ## background hazard rate
    hb_meanlog10 = priorMean(hb_min, hb_max),
    hb_sdlog10 = priorSD(hb_min, hb_max)
  )
  if (model_type == "IT") {
    ### non effect threshold: scale parameter & median of a log-logistic distribution
    priors$alpha_meanlog10 <- priorMean(conc_min, conc_max)
    priors$alpha_sdlog10 <- priorSD(conc_min, conc_max)
    ### shape parameter of a log-logistic distribution
    priors$beta_minlog10 <- beta_minlog10
    priors$beta_maxlog10 <- beta_maxlog10
  } else if (model_type == "SD") {
    ### killing rate parameter: kk
    priors$kk_meanlog10 <- priorMean(kk_min, kk_max) 
    priors$kk_sdlog10 <- priorSD(kk_min, kk_max) 
    ### non effect threshold: z
    priors$z_meanlog10 <- priorMean(conc_min, conc_max)
    priors$z_sdlog10 <- priorSD(conc_min, conc_max) 
  }
  if (is.null(hb_value)) {
    priors$HB_FIXED = 0
    priors$hb_value = -999
  } else{
    priors$HB_FIXED = 1
    priors$hb_value = hb_value
  }
  return(priors)
}


# internal --------------------------------------------------------------------
# Compute priors Mean and SD for lognormal distribution
priorMean <- function(x_min, x_max){
    (log10(x_max) + log10(x_min)) / 2
} 

priorSD <- function(x_min, x_max){
    (log10(x_max) - log10(x_min)) / 4
}

