% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mvgam.R
\name{mvgam}
\alias{mvgam}
\title{Fit a Bayesian Dynamic GAM to Univariate or Multivariate Time Series}
\usage{
mvgam(
  formula,
  trend_formula,
  knots,
  trend_knots,
  trend_model = "None",
  noncentred = FALSE,
  family = poisson(),
  share_obs_params = FALSE,
  data,
  newdata,
  use_lv = FALSE,
  n_lv,
  trend_map,
  priors,
  run_model = TRUE,
  prior_simulation = FALSE,
  residuals = TRUE,
  return_model_data = FALSE,
  backend = getOption("brms.backend", "cmdstanr"),
  algorithm = getOption("brms.algorithm", "sampling"),
  control = list(max_treedepth = 10, adapt_delta = 0.8),
  chains = 4,
  burnin = 500,
  samples = 500,
  thin = 1,
  parallel = TRUE,
  threads = 1,
  save_all_pars = FALSE,
  silent = 1,
  autoformat = TRUE,
  refit = FALSE,
  lfo = FALSE,
  ...
)
}
\arguments{
\item{formula}{A \code{formula} object specifying the GAM observation model formula.
These are exactly like the formula for a GLM except that smooth terms, \code{s()},
\code{te()}, \code{ti()}, \code{t2()}, as well as time-varying \code{dynamic()} terms,
nonparametric \code{gp()} terms and offsets using \code{offset()}, can be added to the
right hand side to specify that the linear predictor depends on smooth
functions of predictors (or linear functionals of these).

In \code{nmix()} family models, the \code{formula} is used to set up a linear predictor
for the detection probability. Details of the formula syntax used by
\pkg{mvgam} can be found in \code{\link{mvgam_formulae}}}

\item{trend_formula}{An optional \code{formula} object specifying the GAM process
model formula. If supplied, a linear predictor will be modelled for the
latent trends to capture process model evolution separately from the
observation model.

\strong{Important notes:}
\itemize{
\item Should not have a response variable specified on the left-hand side
(e.g., \code{~ season + s(year)})
\item Use \code{trend} instead of \code{series} for effects that vary across time series
\item Only available for \code{RW()}, \code{AR()} and \code{VAR()} trend models
\item In \code{nmix()} family models, sets up linear predictor for latent abundance
\item Consider dropping one intercept using \code{- 1} convention to avoid
estimation challenges
}}

\item{knots}{An optional \code{list} containing user specified knot values for
basis construction. For most bases the user simply supplies the knots to be
used, which must match up with the \code{k} value supplied. Different terms can
use different numbers of knots, unless they share a covariate.}

\item{trend_knots}{As for \code{knots} above, this is an optional \code{list} of knot
values for smooth functions within the \code{trend_formula}.}

\item{trend_model}{\code{character} or \code{function} specifying the time series dynamics
for the latent trend.

\strong{Available options:}
\itemize{
\item \code{None}: No latent trend component (GAM component only, like \code{\link[mgcv]{gam}})
\item \code{ZMVN} or \code{ZMVN()}: Zero-Mean Multivariate Normal (Stan only)
\item \code{'RW'} or \code{RW()}: Random Walk
\item \code{'AR1'}, \code{'AR2'}, \code{'AR3'} or \code{AR(p = 1, 2, 3)}: Autoregressive models
\item \code{'CAR1'} or \code{CAR(p = 1)}: Continuous-time AR (Ornstein–Uhlenbeck process)
\item \code{'VAR1'} or \code{VAR()}: Vector Autoregressive (Stan only)
\item \code{'PWlogistic'}, \code{'PWlinear'} or \code{PW()}: Piecewise trends (Stan only)
\item \code{'GP'} or \code{GP()}: Gaussian Process with squared exponential kernel (Stan only)
}

\strong{Additional features:}
\itemize{
\item Moving average and/or correlated process error terms available for most types
(e.g., \code{RW(cor = TRUE)} for multivariate Random Walk)
\item Hierarchical correlations possible for structured data
\item See \link{mvgam_trends} for details and \code{\link[=ZMVN]{ZMVN()}} for examples
}}

\item{noncentred}{\code{logical}. Use non-centred parameterisation for autoregressive
trend models? Can improve efficiency by avoiding degeneracies in latent dynamic
random effects estimation. Benefits vary by model - highly informative data
may perform worse with this option. Available for \code{RW()}, \code{AR()}, \code{CAR()},
or \code{trend = 'None'} with \code{trend_formula}. Not available for moving average
or correlated error models.}

\item{family}{\code{family} specifying the exponential observation family for the series.

\strong{Supported families:}
\itemize{
\item \code{gaussian()}: Real-valued data
\item \code{betar()}: Proportional data on \verb{(0,1)}
\item \code{lognormal()}: Non-negative real-valued data
\item \code{student_t()}: Real-valued data
\item \code{Gamma()}: Non-negative real-valued data
\item \code{bernoulli()}: Binary data
\item \code{poisson()}: Count data (default)
\item \code{nb()}: Overdispersed count data
\item \code{binomial()}: Count data with imperfect detection when number of trials is known
(use \code{cbind()} to bind observations and trials)
\item \code{beta_binomial()}: As \code{binomial()} but allows for overdispersion
\item \code{nmix()}: Count data with imperfect detection when number of trials is unknown
(State-Space N-Mixture model with Poisson latent states and Binomial observations)
}

See \code{\link{mvgam_families}} for more details.}

\item{share_obs_params}{\code{logical}. If \code{TRUE} and the \code{family} has additional
family-specific observation parameters (e.g., variance components, dispersion
parameters), these will be shared across all outcome variables. Useful when
multiple outcomes share properties. Default is \code{FALSE}.}

\item{data}{A \code{dataframe} or \code{list} containing the model response variable
and covariates required by the GAM \code{formula} and optional \code{trend_formula}.

\strong{Required columns for most models:}
\itemize{
\item \code{series}: A \code{factor} index of the series IDs (number of levels should equal
number of unique series labels)
\item \code{time}: \code{numeric} or \code{integer} index of time points. For most dynamic trend
types, time should be measured in discrete, regularly spaced intervals
(i.e., \code{c(1, 2, 3, ...)}). Irregular spacing is allowed for \code{trend_model = CAR(1)},
but zero intervals are adjusted to \code{1e-12} to prevent sampling errors.
}

\strong{Special cases:}
\itemize{
\item Models with hierarchical temporal correlation (e.g., \code{AR(gr = region, subgr = species)})
should NOT include a \code{series} identifier
\item Models without temporal dynamics (\code{trend_model = 'None'} or \code{trend_model = ZMVN()})
don't require a \code{time} variable
}}

\item{newdata}{Optional \code{dataframe} or \code{list} of test data containing the same
variables as in \code{data}. If included, observations in variable \code{y} will be
set to \code{NA} when fitting the model so that posterior simulations can be obtained.}

\item{use_lv}{\code{logical}. If \code{TRUE}, use dynamic factors to estimate series'
latent trends in a reduced dimension format. Only available for \code{RW()},
\code{AR()} and \code{GP()} trend models. Default is \code{FALSE}.
See \code{\link{lv_correlations}} for examples.}

\item{n_lv}{\code{integer} specifying the number of latent dynamic factors to use
if \code{use_lv == TRUE}. Cannot exceed \code{n_series}. Default is
\code{min(2, floor(n_series / 2))}.}

\item{trend_map}{Optional \code{data.frame} specifying which series should depend on
which latent trends. Enables multiple series to depend on the same latent
trend process with different observation processes.

\strong{Required structure:}
\itemize{
\item Column \code{series}: Single unique entry for each series (matching factor levels in data)
\item Column \code{trend}: Integer values indicating which trend each series depends on
}

\strong{Notes:}
\itemize{
\item Sets up latent factor model by enabling \code{use_lv = TRUE}
\item Process model intercept is NOT automatically suppressed
\item Not yet supported for continuous time models (\code{CAR()})
}}

\item{priors}{An optional \code{data.frame} with prior definitions or, preferably,
a vector of \code{brmsprior} objects (see \code{\link[brms]{prior}()}).
See \code{\link[=get_mvgam_priors]{get_mvgam_priors()}} and Details for more information.}

\item{run_model}{\code{logical}. If \code{FALSE}, the model is not fitted but instead
the function returns the model file and the data/initial values needed to
fit the model outside of \code{mvgam}.}

\item{prior_simulation}{\code{logical}. If \code{TRUE}, no observations are fed to the
model, and instead simulations from prior distributions are returned.}

\item{residuals}{\code{logical}. Whether to compute series-level randomized quantile
residuals. Default is \code{TRUE}. Set to \code{FALSE} to save time and reduce object
size (can add later using \link{add_residuals}).}

\item{return_model_data}{\code{logical}. If \code{TRUE}, the list of data needed to fit
the model is returned, along with initial values for smooth and AR parameters,
once the model is fitted. Helpful for users who wish to modify the model file
to add other stochastic elements. Default is \code{FALSE} unless \code{run_model == FALSE}.}

\item{backend}{Character string naming the package for Stan model fitting.
Options are \code{"cmdstanr"} (default) or \code{"rstan"}. Can be set globally via
\code{"brms.backend"} option. See https://mc-stan.org/rstan/ and
https://mc-stan.org/cmdstanr/ for details.}

\item{algorithm}{Character string naming the estimation approach:
\itemize{
\item \code{"sampling"}: MCMC (default)
\item \code{"meanfield"}: Variational inference with factorized normal distributions
\item \code{"fullrank"}: Variational inference with multivariate normal distribution
\item \code{"laplace"}: Laplace approximation (cmdstanr only)
\item \code{"pathfinder"}: Pathfinder algorithm (cmdstanr only)
}

Can be set globally via \code{"brms.algorithm"} option. Limited testing suggests
\code{"meanfield"} performs best among non-MCMC approximations for dynamic GAMs.}

\item{control}{Named \code{list} for controlling sampler behaviour. Valid elements
include \code{max_treedepth}, \code{adapt_delta} and \code{init}.}

\item{chains}{\code{integer} specifying the number of parallel chains for the model.
Ignored for variational inference algorithms.}

\item{burnin}{\code{integer} specifying the number of warmup iterations to tune
sampling algorithms. Ignored for variational inference algorithms.}

\item{samples}{\code{integer} specifying the number of post-warmup iterations for
sampling the posterior distribution.}

\item{thin}{Thinning interval for monitors. Ignored for variational inference
algorithms.}

\item{parallel}{\code{logical} specifying whether to use multiple cores for parallel
MCMC simulation. If \code{TRUE}, uses \code{min(c(chains, parallel::detectCores() - 1))} cores.}

\item{threads}{\code{integer}. Experimental option for within-chain parallelisation
in Stan using \code{reduce_sum}. Recommended only for experienced Stan users with
slow models. Currently works for all families except \code{nmix()} and when using
Cmdstan backend.}

\item{save_all_pars}{\code{logical}. Save draws from all variables defined in Stan's
\code{parameters} block. Default is \code{FALSE}.}

\item{silent}{Verbosity level between \code{0} and \code{2}. If \code{1} (default), most
informational messages are suppressed. If \code{2}, even more messages are
suppressed. Sampling progress is still printed - set \code{refresh = 0} to
disable. For \code{backend = "rstan"}, also set \code{open_progress = FALSE} to
prevent additional progress bars.}

\item{autoformat}{\code{logical}. Use \code{stanc} parser to automatically format Stan
code and check for deprecations. For development purposes - leave as \code{TRUE}.}

\item{refit}{\code{logical}. Indicates whether this is a refit called using
\code{\link[=update.mvgam]{update.mvgam()}}. Users should leave as \code{FALSE}.}

\item{lfo}{\code{logical}. Indicates whether this is part of \link{lfo_cv.mvgam} call.
Returns lighter model version for speed. Users should leave as \code{FALSE}.}

\item{...}{Further arguments passed to Stan:
\itemize{
\item For \code{backend = "rstan"}: passed to \code{\link[rstan]{sampling}()} or
\code{\link[rstan]{vb}()}
\item For \code{backend = "cmdstanr"}: passed to \code{cmdstanr::sample},
\code{cmdstanr::variational}, \code{cmdstanr::laplace} or \code{cmdstanr::pathfinder} methods
}}
}
\value{
A \code{list} object of class \code{mvgam} containing model output, the text
representation of the model file, the mgcv model output (for easily generating
simulations at unsampled covariate values), Dunn-Smyth residuals for each
series and key information needed for other functions in the package. See
\code{\link{mvgam-class}} for details. Use \code{methods(class = "mvgam")} for an
overview on available methods.
}
\description{
This function estimates the posterior distribution for Generalised Additive
Models (GAMs) that can include smooth spline functions, specified in the GAM
formula, as well as latent temporal processes, specified by \code{trend_model}.

Further modelling options include State-Space representations to allow covariates
and dynamic processes to occur on the latent 'State' level while also capturing
observation-level effects. Prior specifications are flexible and explicitly
encourage users to apply prior distributions that actually reflect their beliefs.

In addition, model fits can easily be assessed and compared with posterior
predictive checks, forecast comparisons and leave-one-out / leave-future-out
cross-validation.
}
\details{
Dynamic GAMs are useful when we wish to predict future values from time series
that show temporal dependence but we do not want to rely on extrapolating from
a smooth term (which can sometimes lead to unpredictable and unrealistic behaviours).
In addition, smooths can often try to wiggle excessively to capture any
autocorrelation that is present in a time series, which exacerbates the problem
of forecasting ahead.

As GAMs are very naturally viewed through a Bayesian lens, and we often must
model time series that show complex distributional features and missing data,
parameters for \pkg{mvgam} models are estimated in a Bayesian framework using
Markov Chain Monte Carlo by default.

\strong{Getting Started Resources:}
\itemize{
\item General overview: \code{vignette("mvgam_overview")} and \code{vignette("data_in_mvgam")}
\item Full list of vignettes: \code{vignette(package = "mvgam")}
\item Real-world examples: \code{\link{mvgam_use_cases}}
\item Quick reference: \href{https://github.com/nicholasjclark/mvgam/raw/master/misc/mvgam_cheatsheet.pdf}{mvgam cheatsheet}
}
}
\section{Model Specification Details}{


\strong{Formula Syntax:} Details of the formula syntax used by \pkg{mvgam} can be
found in \code{\link{mvgam_formulae}}. Note that it is possible to supply an
empty formula where there are no predictors or intercepts in the observation
model (i.e. \code{y ~ 0} or \code{y ~ -1}). In this case, an intercept-only observation
model will be set up but the intercept coefficient will be fixed at zero. This
can be handy if you wish to fit pure State-Space models where the variation in
the dynamic trend controls the average expectation, and/or where intercepts are
non-identifiable (as in piecewise trends).

\strong{Families and Link Functions:} Details of families supported by \pkg{mvgam}
can be found in \code{\link{mvgam_families}}.

\strong{Trend Models:} Details of latent error process models supported by \pkg{mvgam}
can be found in \code{\link{mvgam_trends}}.
}

\section{Prior Specifications}{

Default priors for intercepts and any variance parameters are chosen to be
vaguely informative, but these should always be checked by the user. Prior
distributions for most important model parameters can be altered (see
\code{\link[=get_mvgam_priors]{get_mvgam_priors()}} for details). Note that latent trends are estimated on
the link scale so choose priors accordingly.

However more control over the model specification can be accomplished by setting
\code{run_model = FALSE} and then editing the model code (found in the
\code{model_file} slot in the returned object) before running the model using either
\pkg{rstan} or \pkg{cmdstanr}. This is encouraged for complex modelling tasks.

\strong{Important:} No priors are formally checked to ensure they are in the right
syntax so it is up to the user to ensure these are correct.
}

\section{Model Components}{


\strong{Random Effects:} For any smooth terms using the random effect basis
(\code{\link[mgcv]{smooth.construct.re.smooth.spec}}), a non-centred
parameterisation is automatically employed to avoid degeneracies that are common
in hierarchical models. Note however that centred versions may perform better
for series that are particularly informative, so as with any foray into Bayesian
modelling, it is worth building an understanding of the model's assumptions and
limitations by following a principled workflow. Also note that models are
parameterised using \code{drop.unused.levels = FALSE} in \code{\link[mgcv]{jagam}}
to ensure predictions can be made for all levels of the supplied factor variable.

\strong{Observation Level Parameters:} When more than one series is included in
\code{data} and an observation family that contains more than one parameter is
used, additional observation family parameters (i.e. \code{phi} for \code{nb()} or \code{sigma}
for \code{gaussian()}) are by default estimated independently for each series. But if
you wish for the series to share the same observation parameters, set
\code{share_obs_params = TRUE}.
}

\section{Model Diagnostics}{


\strong{Residuals:} For each series, randomized quantile (i.e. Dunn-Smyth) residuals
are calculated for inspecting model diagnostics. If the fitted model is
appropriate then Dunn-Smyth residuals will be standard normal in distribution
and no autocorrelation will be evident. When a particular observation is missing,
the residual is calculated by comparing independent draws from the model's
posterior distribution.
}

\section{Computational Backend}{


\strong{Using Stan:} \pkg{mvgam} is primarily designed to use Hamiltonian Monte Carlo
for parameter estimation via the software \code{Stan} (using either the \code{cmdstanr}
or \code{rstan} interface). There are great advantages when using \code{Stan} over Gibbs /
Metropolis Hastings samplers, which includes the option to estimate nonlinear
effects via \href{https://arxiv.org/abs/2004.11408}{Hilbert space approximate Gaussian Processes},
the availability of a variety of inference algorithms (i.e. variational inference,
laplacian inference etc...) and \href{https://www.tandfonline.com/doi/full/10.1080/10618600.2022.2079648}{capabilities to enforce stationarity for complex Vector Autoregressions}.

Because of the many advantages of \code{Stan} over \code{JAGS}, \strong{further development of
the package will only be applied to \code{Stan}}. This includes the planned addition
of more response distributions, plans to handle zero-inflation, and plans to
incorporate a greater variety of trend models. Users are strongly encouraged to
opt for \code{Stan} over \code{JAGS} in any proceeding workflows.
}

\section{Recommended Workflow}{


\strong{How to Start:} The \href{https://github.com/nicholasjclark/mvgam/raw/master/misc/mvgam_cheatsheet.pdf}{\code{mvgam} cheatsheet}
is a good starting place if you are just learning to use the package. It gives
an overview of the package's key functions and objects, as well as providing a
reasonable workflow that new users can follow.

\strong{Recommended Steps:}
\enumerate{
\item \strong{Data Preparation:} Check that your data are in a suitable tidy format for
\pkg{mvgam} modeling (see the \href{https://nicholasjclark.github.io/mvgam/articles/data_in_mvgam.html}{data formatting vignette}
for guidance)
\item \strong{Data Exploration:} Inspect features of the data using \code{\link{plot_mvgam_series}}.
Now is also a good time to familiarise yourself with the package's example
workflows that are detailed in the vignettes:
\itemize{
\item \href{https://nicholasjclark.github.io/mvgam/articles/mvgam_overview.html}{Getting started vignette}
\item \href{https://nicholasjclark.github.io/mvgam/articles/shared_states.html}{Shared latent states vignette}
\item \href{https://nicholasjclark.github.io/mvgam/articles/time_varying_effects.html}{Time-varying effects vignette}
\item \href{https://nicholasjclark.github.io/mvgam/articles/trend_formulas.html}{State-Space models vignette}
\item \href{https://nicholasjclark.github.io/mvgam/articles/nmixtures.html}{"Fitting N-mixture models in \code{mvgam}"}
\item \href{https://nicholasjclark.github.io/mvgam/reference/jsdgam.html}{"Joint Species Distribution Models in \code{mvgam}"}
\item \href{https://ecogambler.netlify.app/blog/time-varying-seasonality/}{"Incorporating time-varying seasonality in forecast models"}
\item \href{https://ecogambler.netlify.app/blog/autocorrelated-gams/}{"Temporal autocorrelation in GAMs and the \code{mvgam} package"}
}
\item \strong{Model Structure:} Carefully think about how to structure linear predictor
effects (i.e. smooth terms using \code{\link[mgcv]{s}()}, \code{\link[mgcv]{te}()}
or \code{\link[mgcv]{ti}()}, GPs using \code{\link[brms]{gp}()}, dynamic
time-varying effects using \code{\link[=dynamic]{dynamic()}}, and parametric terms), latent temporal
trend components (see \code{\link{mvgam_trends}}) and the appropriate
observation family (see \code{\link{mvgam_families}}). Use \code{\link[=get_mvgam_priors]{get_mvgam_priors()}}
to see default prior distributions for stochastic parameters.
\item \strong{Prior Specification:} Change default priors using appropriate prior knowledge
(see \code{\link[brms]{prior}()}). When using State-Space models with a
\code{trend_formula}, pay particular attention to priors for any variance parameters
such as process errors and observation errors. Default priors on these parameters
are chosen to be vaguely informative and to avoid zero (using Inverse Gamma
priors), but more informative priors will often help with model efficiency
and convergence.
\item \strong{Model Fitting:} Fit the model using either Hamiltonian Monte Carlo or an
approximation algorithm (i.e. change the \code{backend} argument) and use
\code{\link[=summary.mvgam]{summary.mvgam()}}, \code{\link[=conditional_effects.mvgam]{conditional_effects.mvgam()}}, \code{\link[=mcmc_plot.mvgam]{mcmc_plot.mvgam()}},
\code{\link[=pp_check.mvgam]{pp_check.mvgam()}}, \code{\link[=pairs.mvgam]{pairs.mvgam()}} and \code{\link[=plot.mvgam]{plot.mvgam()}} to inspect /
interrogate the model.
\item \strong{Model Comparison:} Update the model as needed and use \code{\link[=loo_compare.mvgam]{loo_compare.mvgam()}}
for in-sample model comparisons, or alternatively use \code{\link[=forecast.mvgam]{forecast.mvgam()}},
\code{\link[=lfo_cv.mvgam]{lfo_cv.mvgam()}} and \code{\link[=score.mvgam_forecast]{score.mvgam_forecast()}} to compare models based on
out-of-sample forecasts (see the \href{https://nicholasjclark.github.io/mvgam/articles/forecast_evaluation.html}{forecast evaluation vignette}
for guidance).
\item \strong{Inference and Prediction:} When satisfied with the model structure, use
\code{\link[=predict.mvgam]{predict.mvgam()}}, \code{\link[marginaleffects]{plot_predictions}()} and/or
\code{\link[marginaleffects]{plot_slopes}()} for more targeted simulation-based
inferences (see \href{https://ecogambler.netlify.app/blog/interpreting-gams/}{"How to interpret and report nonlinear effects from Generalized Additive Models"}
for some guidance on interpreting GAMs). For time series models, use
\code{\link[=hindcast.mvgam]{hindcast.mvgam()}}, \code{\link[=fitted.mvgam]{fitted.mvgam()}}, \code{\link[=augment.mvgam]{augment.mvgam()}} and \code{\link[=forecast.mvgam]{forecast.mvgam()}}
to inspect posterior hindcast / forecast distributions.
\item \strong{Documentation:} Use \code{\link[=how_to_cite]{how_to_cite()}} to obtain a scaffold methods section
(with full references) to begin describing this model in scientific publications.
}
}

\examples{
\donttest{
# =============================================================================
# Basic Multi-Series Time Series Modeling
# =============================================================================

# Simulate three time series that have shared seasonal dynamics,
# independent AR(1) trends, and Poisson observations
set.seed(0)
dat <- sim_mvgam(
  T = 80,
  n_series = 3,
  mu = 2,
  trend_model = AR(p = 1),
  prop_missing = 0.1,
  prop_trend = 0.6
)

# Plot key summary statistics for a single series
plot_mvgam_series(data = dat$data_train, series = 1)

# Plot all series together
plot_mvgam_series(data = dat$data_train, series = "all")

# Formulate a model using Stan where series share a cyclic smooth for
# seasonality and each series has an independent AR1 temporal process.
# Note that 'noncentred = TRUE' will likely give performance gains.
# Set run_model = FALSE to inspect the returned objects
mod1 <- mvgam(
  formula = y ~ s(season, bs = "cc", k = 6),
  data = dat$data_train,
  trend_model = AR(),
  family = poisson(),
  noncentred = TRUE,
  run_model = FALSE
)

# View the model code in Stan language
stancode(mod1)

# View the data objects needed to fit the model in Stan
sdata1 <- standata(mod1)
str(sdata1)

# Now fit the model
mod1 <- mvgam(
  formula = y ~ s(season, bs = "cc", k = 6),
  data = dat$data_train,
  trend_model = AR(),
  family = poisson(),
  noncentred = TRUE,
  chains = 2,
  silent = 2
)

# Extract the model summary
summary(mod1)

# Plot the historical trend and hindcast distributions for one series
hc_trend <- hindcast(mod1, type = "trend")
plot(hc_trend)

hc_predicted <- hindcast(mod1, type = "response")
plot(hc_predicted)

# Residual diagnostics
plot(mod1, type = "residuals", series = 1)
resids <- residuals(mod1)
str(resids)

# Fitted values and residuals can be added directly to the training data
augment(mod1)

# Compute the forecast using covariate information in data_test
fc <- forecast(mod1, newdata = dat$data_test)
str(fc)
fc_summary <- summary(fc)
head(fc_summary, 12)
plot(fc)

# Plot the estimated seasonal smooth function
plot(mod1, type = "smooths")

# Plot estimated first derivatives of the smooth
plot(mod1, type = "smooths", derivatives = TRUE)

# Plot partial residuals of the smooth
plot(mod1, type = "smooths", residuals = TRUE)

# Plot posterior realisations for the smooth
plot(mod1, type = "smooths", realisations = TRUE)

# Plot conditional response predictions using marginaleffects
conditional_effects(mod1)
plot_predictions(mod1, condition = "season", points = 0.5)

# Generate posterior predictive checks using bayesplot
pp_check(mod1)

# Extract observation model beta coefficient draws as a data.frame
beta_draws_df <- as.data.frame(mod1, variable = "betas")
head(beta_draws_df)
str(beta_draws_df)

# Investigate model fit
mc.cores.def <- getOption("mc.cores")
options(mc.cores = 1)
loo(mod1)
options(mc.cores = mc.cores.def)


# =============================================================================
# Vector Autoregressive (VAR) Models
# =============================================================================

# Fit a model to the portal time series that uses a latent
# Vector Autoregression of order 1
mod <- mvgam(
  formula = captures ~ -1,
  trend_formula = ~ trend,
  trend_model = VAR(cor = TRUE),
  family = poisson(),
  data = portal_data,
  chains = 2,
  silent = 2
)

# Plot the autoregressive coefficient distributions;
# use 'dir = "v"' to arrange the order of facets correctly
mcmc_plot(
  mod,
  variable = 'A',
  regex = TRUE,
  type = 'hist',
  facet_args = list(dir = 'v')
)

# Plot the process error variance-covariance matrix in the same way
mcmc_plot(
  mod,
  variable = 'Sigma',
  regex = TRUE,
  type = 'hist',
  facet_args = list(dir = 'v')
)

# Calculate Generalized Impulse Response Functions for each series
irfs <- irf(
  mod,
  h = 12,
  cumulative = FALSE
)

# Plot some of them
plot(irfs, series = 1)
plot(irfs, series = 2)

# Calculate forecast error variance decompositions for each series
fevds <- fevd(mod, h = 12)

# Plot median contributions to forecast error variance
plot(fevds)


# =============================================================================
# Dynamic Factor Models
# =============================================================================

# Now fit a model that uses two RW dynamic factors to model
# the temporal dynamics of the four rodent species
mod <- mvgam(
  captures ~ series,
  trend_model = RW(),
  use_lv = TRUE,
  n_lv = 2,
  data = portal_data,
  chains = 2,
  silent = 2
)

# Plot the factors
plot(mod, type = 'factors')

# Plot the hindcast distributions
hcs <- hindcast(mod)
plot(hcs, series = 1)
plot(hcs, series = 2)
plot(hcs, series = 3)
plot(hcs, series = 4)

# Use residual_cor() to calculate temporal correlations among the series
# based on the factor loadings
lvcors <- residual_cor(mod)
names(lvcors)
lvcors$cor

# For those correlations whose credible intervals did not include
# zero, plot them as a correlation matrix (all other correlations
# are shown as zero on this plot)
plot(lvcors, cluster = TRUE)


# =============================================================================
# Shared Latent Trends with Custom Trend Mapping
# =============================================================================

# Example of supplying a trend_map so that some series can share
# latent trend processes
sim <- sim_mvgam(n_series = 3)
mod_data <- sim$data_train

# Here, we specify only two latent trends; series 1 and 2 share a trend,
# while series 3 has its own unique latent trend
trend_map <- data.frame(
  series = unique(mod_data$series),
  trend = c(1, 1, 2)
)

# Fit the model using AR1 trends
mod <- mvgam(
  formula = y ~ s(season, bs = "cc", k = 6),
  trend_map = trend_map,
  trend_model = AR(),
  data = mod_data,
  return_model_data = TRUE,
  chains = 2,
  silent = 2
)

# The mapping matrix is now supplied as data to the model in the 'Z' element
mod$model_data$Z

# The first two series share an identical latent trend; the third is different
plot(residual_cor(mod))
plot(mod, type = "trend", series = 1)
plot(mod, type = "trend", series = 2)
plot(mod, type = "trend", series = 3)


# =============================================================================
# Time-Varying (Dynamic) Coefficients
# =============================================================================

# Example of how to use dynamic coefficients
# Simulate a time-varying coefficient for the effect of temperature
set.seed(123)
N <- 200
beta_temp <- vector(length = N)
beta_temp[1] <- 0.4
for (i in 2:N) {
  beta_temp[i] <- rnorm(1, mean = beta_temp[i - 1] - 0.0025, sd = 0.05)
}
plot(beta_temp)

# Simulate a covariate called 'temp'
temp <- rnorm(N, sd = 1)

# Simulate some noisy Gaussian observations
out <- rnorm(N,
  mean = 4 + beta_temp * temp,
  sd = 0.5
)

# Gather necessary data into a data.frame; split into training / testing
data <- data.frame(out, temp, time = seq_along(temp))
data_train <- data[1:180, ]
data_test <- data[181:200, ]

# Fit the model using the dynamic() function
mod <- mvgam(
  formula = out ~ dynamic(
    temp,
    scale = FALSE,
    k = 40
  ),
  family = gaussian(),
  data = data_train,
  newdata = data_test,
  chains = 2,
  silent = 2
)

# Inspect the model summary, forecast and time-varying coefficient distribution
summary(mod)
plot(mod, type = "smooths")
fc <- forecast(mod, newdata = data_test)
plot(fc)

# Propagating the smooth term shows how the coefficient is expected to evolve
plot_mvgam_smooth(mod, smooth = 1, newdata = data)
abline(v = 180, lty = "dashed", lwd = 2)
points(beta_temp, pch = 16)


# =============================================================================
# Working with Offset Terms
# =============================================================================

# Example showing how to incorporate an offset; simulate some count data
# with different means per series
set.seed(100)
dat <- sim_mvgam(
  prop_trend = 0,
  mu = c(0, 2, 2),
  seasonality = "hierarchical"
)

# Add offset terms to the training and testing data
dat$data_train$offset <- 0.5 * as.numeric(dat$data_train$series)
dat$data_test$offset <- 0.5 * as.numeric(dat$data_test$series)

# Fit a model that includes the offset in the linear predictor as well as
# hierarchical seasonal smooths
mod <- mvgam(
  formula = y ~ offset(offset) +
    s(series, bs = "re") +
    s(season, bs = "cc") +
    s(season, by = series, m = 1, k = 5),
  data = dat$data_train,
  chains = 2,
  silent = 2
)

# Inspect the model file to see the modification to the linear predictor (eta)
stancode(mod)

# Forecasts for the first two series will differ in magnitude
fc <- forecast(mod, newdata = dat$data_test)
plot(fc, series = 1, ylim = c(0, 75))
plot(fc, series = 2, ylim = c(0, 75))

# Changing the offset for the testing data should lead to changes in
# the forecast
dat$data_test$offset <- dat$data_test$offset - 2
fc <- forecast(mod, newdata = dat$data_test)
plot(fc)

# Relative Risks can be computed by fixing the offset to the same value
# for each series
dat$data_test$offset <- rep(1, NROW(dat$data_test))
preds_rr <- predict(mod,
  type = "link",
  newdata = dat$data_test,
  summary = FALSE
)
series1_inds <- which(dat$data_test$series == "series_1")
series2_inds <- which(dat$data_test$series == "series_2")

# Relative Risks are now more comparable among series
layout(matrix(1:2, ncol = 2))
plot(preds_rr[1, series1_inds],
  type = "l", col = "grey75",
  ylim = range(preds_rr),
  ylab = "Series1 Relative Risk", xlab = "Time"
)
for (i in 2:50) {
  lines(preds_rr[i, series1_inds], col = "grey75")
}

plot(preds_rr[1, series2_inds],
  type = "l", col = "darkred",
  ylim = range(preds_rr),
  ylab = "Series2 Relative Risk", xlab = "Time"
)
for (i in 2:50) {
  lines(preds_rr[i, series2_inds], col = "darkred")
}
layout(1)


# =============================================================================
# Binomial Family Models
# =============================================================================

# Example showcasing how cbind() is needed for Binomial observations
# Simulate two time series of Binomial trials
trials <- sample(c(20:25), 50, replace = TRUE)
x <- rnorm(50)
detprob1 <- plogis(-0.5 + 0.9 * x)
detprob2 <- plogis(-0.1 - 0.7 * x)
dat <- rbind(
  data.frame(
    y = rbinom(n = 50, size = trials, prob = detprob1),
    time = 1:50,
    series = "series1",
    x = x,
    ntrials = trials
  ),
  data.frame(
    y = rbinom(n = 50, size = trials, prob = detprob2),
    time = 1:50,
    series = "series2",
    x = x,
    ntrials = trials
  )
)
dat <- dplyr::mutate(dat, series = as.factor(series))
dat <- dplyr::arrange(dat, time, series)
plot_mvgam_series(data = dat, series = "all")

# Fit a model using the binomial() family; must specify observations
# and number of trials in the cbind() wrapper
mod <- mvgam(
  formula = cbind(y, ntrials) ~ series + s(x, by = series),
  family = binomial(),
  data = dat,
  chains = 2,
  silent = 2
)
summary(mod)
pp_check(mod,
  type = "bars_grouped",
  group = "series", ndraws = 50
)
pp_check(mod,
  type = "ecdf_overlay_grouped",
  group = "series", ndraws = 50
)
conditional_effects(mod, type = "link")

# To view predictions on the probability scale,
# use ntrials = 1 in datagrid()
plot_predictions(
  mod,
  by = c('x', 'series'),
  newdata = datagrid(
    x = runif(100, -2, 2),
    series = unique,
    ntrials = 1
  ),
  type = 'expected'
)

\dontshow{
# For R CMD check: make sure any open connections are closed afterward
closeAllConnections()
}
}
}
\references{
Nicholas J Clark & Konstans Wells (2023). Dynamic generalised additive models
(DGAMs) for forecasting discrete ecological time series. Methods in Ecology and
Evolution. 14:3, 771-784.

Nicholas J Clark, SK Morgan Ernest, Henry Senyondo, Juniper Simonis, Ethan P White,
Glenda M Yenni, KANK Karunarathna (2025). Beyond single-species models: leveraging
multispecies forecasts to navigate the dynamics of ecological predictability.
PeerJ. 13:e18929 https://doi.org/10.7717/peerj.18929
}
\seealso{
\code{\link[mgcv]{jagam}()}, \code{\link[mgcv]{gam}()},
\code{\link[mgcv]{gam.models}}, \code{\link[=get_mvgam_priors]{get_mvgam_priors()}}, \code{\link[=jsdgam]{jsdgam()}},
\code{\link[=hindcast.mvgam]{hindcast.mvgam()}}, \code{\link[=forecast.mvgam]{forecast.mvgam()}}, \code{\link[=predict.mvgam]{predict.mvgam()}}
}
\author{
Nicholas J Clark
}
