% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/M_statistic.R
\name{M_rand_perm}
\alias{M_rand_perm}
\title{Calculate M statistics after random permutations}
\usage{
M_rand_perm(
  trait_dist = NULL,
  phy = NULL,
  reps = 999,
  auto_multi2di = TRUE,
  cores = 1
)
}
\arguments{
\item{trait_dist}{A distance object of class \code{matrix} or \code{dist}.
Its row and column names should match the tip labels of the phylogenetic tree (\code{phy}).
The functions \code{\link[=gower_dist]{gower_dist()}} and \code{\link[cluster:daisy]{cluster::daisy()}} can be used to calculate distances using trait data.}

\item{phy}{A phylogenetic tree of class \code{phylo}.}

\item{reps}{An integer. The number of random permutations.}

\item{auto_multi2di}{A logical switch, \code{TRUE} or \code{FALSE}. Default is \code{TRUE},
then function \code{\link[ape:multi2di]{ape::multi2di()}} in \code{ape} package will be called to make the phylogeney (tree)
be dichotomous if the tree (\code{phy}) contains some polytomies.}

\item{cores}{Number of cores to be used in parallel processing.
Default is 1, indicating no parallel computation is performed.
If set to 0, parallel computation is executed using \code{parallel::detectCores() - 1} number of cores.}
}
\value{
A list object containing two components.
Component \verb{$permuted} is the vector of M values obtained after random permutation for \code{reps} times;
component \verb{$observed} is the value of M statistic obtained from the original input data.
}
\description{
\code{M_rand_perm} calculates M statistic for trait(s) after randomly permuting the species names or tip labels in phylogeny.
The M statistic is a unified method for detecting phylogenetic signals in continuous traits,
discrete traits, and multi-trait combinations.
Blomberg and Garland (2002) provided a widely accepted statistical definition of
the phylogenetic signal, which is the "tendency for related species to resemble
each other more than they resemble species drawn at random from the tree".
The M statistic strictly adheres to the definition of phylogenetic signal,
formulating an index and developing a method of testing in strict accordance
with the definition, instead of relying on correlation analysis or evolutionary models.
The novel method equivalently expressed the textual definition of the phylogenetic signal
as an inequality equation of the phylogenetic and trait distances and constructed the M statistic.
The M statistic implemented in this package is based on the methodology described in Yao and Yuan (2025) \url{doi:10.1002/ece3.71106}. If you use this method in your research, please cite the paper.
}
\examples{
data("turtles")
# Continuous trait
trait_df <- data.frame(M1 = turtles$traits$M1, row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df)
M_rand_perm(trait_dist, turtles$phylo, reps = 99) # reps=999 better

# Nominal discrete trait
trait_df <- data.frame(B1 = turtles$traits$B1, row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df, type = list(factor = 1))
M_rand_perm(trait_dist, turtles$phylo, reps = 99) # reps=999 better

# Ordinal discrete trait
trait_df <- data.frame(CS1 = turtles$traits$CS1, row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df, type = list(ordered = 1))
M_rand_perm(trait_dist, turtles$phylo, reps = 99) # reps=999 better

# Multi-trait Combinations
trait_df <- data.frame(turtles$traits[, c("M1", "M2", "M3", "M4", "M5")],
                       row.names = turtles$traits$specie)
trait_dist <- gower_dist(x = trait_df, type = list(factor = c("M4", "M5")))
M_rand_perm(trait_dist, turtles$phylo, reps = 99) # reps=999 better

}
\references{
Blomberg, S.P. & Garland, T., Jr (2002) Tempo and mode in evolution: phylogenetic inertia, adaptation and comparative methods. Journal of Evolutionary Biology, 15(6): 899-910.

Yao, L. and Yuan, Y. (2025), A Unified Method for Detecting Phylogenetic Signals in Continuous, Discrete, and Multiple Trait Combinations. Ecology and Evolution, 15: e71106.
}
\seealso{
\code{\link[=M_stat]{M_stat()}} \code{\link[=phylosignal_M]{phylosignal_M()}}
}
