% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/marxan_problem.R
\name{marxan_problem}
\alias{marxan_problem}
\alias{marxan_problem.default}
\alias{marxan_problem.data.frame}
\alias{marxan_problem.character}
\title{\emph{Marxan} conservation problem}
\usage{
marxan_problem(x, ...)

\method{marxan_problem}{default}(x, ...)

\method{marxan_problem}{data.frame}(x, spec, puvspr, bound = NULL, blm = 0, symmetric = TRUE, ...)

\method{marxan_problem}{character}(x, ...)
}
\arguments{
\item{x}{\code{character} file path for a \emph{Marxan} input file (typically
called \code{"input.dat"}), or \code{data.frame} containing planning unit
data (typically called \code{"pu.dat"}). If the argument to \code{x} is a
\code{data.frame}, then each row corresponds to a different planning unit,
and it must have the following columns:

\describe{

\item{id}{\code{integer} unique identifier for each planning unit.
These identifiers are used in the argument to \code{puvspr}.}

\item{cost}{\code{numeric} cost of each planning unit.}

\item{status}{\code{integer} indicating if each planning unit
should not be locked in the solution (0) or if it should be locked in
(2) or locked out (3) of the solution. Although \emph{Marxan} allows
planning units to be selected in the initial solution (using values of
1), these values have no effect here. This column is optional.}

}}

\item{...}{not used.}

\item{spec}{\code{data.frame} containing information on the features.
The argument to \code{spec} must follow the conventions used by
\emph{Marxan} for the species data file (conventionally called
\code{"spec.dat"}). Each row corresponds to a different feature and
each column corresponds to different information about the features. It
must contain the columns listed below. Note that the argument to
\code{spec} must contain at least one column named \code{"prop"} or
\code{"amount"}---\strong{but not both columns with both of these
names}---to specify the target for each feature.

\describe{

\item{id}{\code{integer} unique identifier for each feature
These identifiers are used in the argument to \code{puvspr}.}

\item{name}{\code{character} name for each feature.}

\item{prop}{\code{numeric} relative target for each feature
(optional).}'

\item{amount}{\code{numeric} absolute target for each
feature (optional).}

}}

\item{puvspr}{\code{data.frame} containing information on the amount of
each feature in each planning unit. The argument to
\code{puvspr} must follow the conventions used in the \emph{Marxan} input
data file (conventionally called \code{"puvspr.dat"}). It must contain the
following columns:

\describe{

\item{pu}{\code{integer} planning unit identifier.}

\item{species}{\code{integer} feature identifier.}

\item{amount}{\code{numeric} amount of the feature in the
planning unit.}

}}

\item{bound}{\code{NULL} object indicating that no boundary data
is required for the conservation planning problem, or a \code{data.frame}
containing information on the planning units' boundaries. The argument to
\code{bound} must follow the conventions used in the \emph{Marxan} input
data file (conventionally called \code{"bound.dat"}). It must contain the
following columns:

\describe{

\item{id1}{\code{integer} planning unit identifier.}

\item{id2}{\code{integer} planning unit identifier.}

\item{boundary}{\code{numeric} length of shared boundary
between the planning units identified in the previous two columns.}

}}

\item{blm}{\code{numeric} boundary length modifier. This argument only
has an effect when argument to \code{x} is a \code{data.frame}. The
default argument is zero.}

\item{symmetric}{\code{logical} does the boundary data (i.e., argument to
\code{bound}) describe symmetric relationships between planning units?
If the boundary data contain asymmetric connectivity data,
this parameter should be set to \code{FALSE}.
Defaults to \code{TRUE}.}
}
\value{
A \code{\link[=problem]{problem()}} object.
}
\description{
Create a conservation planning \code{\link[=problem]{problem()}} following the
mathematical formulations used in \emph{Marxan} (detailed in Beyer
\emph{et al.} 2016). Note that these problems are solved using
exact algorithms and not simulated annealing (i.e., the \emph{Marxan} software).
Please note that the \pkg{vroom} package is required to import \emph{Marxan} data
files.
}
\details{
This function is provided as a convenient interface for solving
\emph{Marxan} problems using the \pkg{prioritizr} package. Note that this
function does not support all of the functionality provided by the
\emph{Marxan} software. In particular, only the following parameters
supported: \code{"INPUTDIR"}, \verb{"SPECNAME}", \code{"PUNAME"}, \verb{"PUVSPRNAME", }"BOUNDNAME"\verb{, }"BLM"\verb{, and }"ASYMMETRICCONNECTIVITY"\verb{. Additionally, for the species data (i.e., argument to }spec\verb{), only the }"id"\verb{, }"name"\verb{, }"prop"\verb{, and }"amount"` columns are considered.
}
\section{Notes}{

In previous versions, this function could not accommodate asymmetric
connectivity data. It has now been updated to handle asymmetric connectivity
data.
}

\examples{
# create Marxan problem using Marxan input file
# (note this example requires the vroom package to be installed)
\dontrun{
input_file <- system.file("extdata/marxan/input.dat", package = "prioritizr")
p1 <-
  marxan_problem(input_file) \%>\%
  add_default_solver(verbose = FALSE)

# solve problem
s1 <- solve(p1)

# print solution
head(s1)

# create Marxan problem using data.frames that have been loaded into R
# (note this example also requires the vroom package to be installed)
## load in planning unit data
pu_path <- system.file("extdata/marxan/input/pu.dat", package = "prioritizr")
pu_dat <- vroom::vroom(pu_path)
head(pu_dat)

## load in feature data
spec_path <- system.file(
  "extdata/marxan/input/spec.dat", package = "prioritizr"
)
spec_dat <- vroom::vroom(spec_path)
head(spec_dat)

## load in planning unit vs feature data
puvspr_path <- system.file(
  "extdata/marxan/input/puvspr.dat", package = "prioritizr"
)
puvspr_dat <- vroom::vroom(puvspr_path)
head(puvspr_dat)

## load in the boundary data
bound_path <- system.file(
  "extdata/marxan/input/bound.dat", package = "prioritizr"
)
bound_dat <- vroom::vroom(bound_path)
head(bound_dat)

# create problem without the boundary data
p2 <-
  marxan_problem(pu_dat, spec_dat, puvspr_dat) \%>\%
  add_default_solver(verbose = FALSE)

# solve problem
s2 <- solve(p2)

# print solution
head(s2)

# create problem with the boundary data and a boundary length modifier
# set to 5
p3 <-
  marxan_problem(pu_dat, spec_dat, puvspr_dat, bound_dat, blm = 5) \%>\%
  add_default_solver(verbose = FALSE)

# solve problem
s3 <- solve(p3)

# print solution
head(s3)
}
}
\references{
Ball IR, Possingham HP, and Watts M (2009) \emph{Marxan and relatives:
Software for spatial conservation prioritisation} in Spatial conservation
prioritisation: Quantitative methods and computational tools. Eds Moilanen
A, Wilson KA, and Possingham HP. Oxford University Press, Oxford, UK.

Beyer HL, Dujardin Y, Watts ME, and Possingham HP (2016) Solving
conservation planning problems with integer linear programming.
\emph{Ecological Modelling}, 228: 14--22.

Serra N, Kockel A, Game ET, Grantham H, Possingham HP, and McGowan J (2020)
Marxan User Manual: For Marxan version 2.43 and above. The Nature Conservancy
(TNC), Arlington, Virginia, United States and Pacific Marine Analysis and
Research Association (PacMARA), Victoria, British Columbia, Canada.
}
\seealso{
For more information on the correct format for
for \emph{Marxan} input data, see the
\href{https://marxansolutions.org}{official \emph{Marxan} website}, Ball
\emph{et al.} (2009), Serra \emph{et al.} (2020).
}
