% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/read_caim_raw.R
\name{read_caim_raw}
\alias{read_caim_raw}
\title{Read a canopy image from a raw file}
\usage{
read_caim_raw(path, only_blue = FALSE, offset_value = NULL)
}
\arguments{
\item{path}{character vector of length one. Path to a file with raw data
(including file extension).}

\item{only_blue}{logical vector of length one. If \code{TRUE}, return only the blue/cyan band.}

\item{offset_value}{numeric vector of length one. Optional black level offsets to replace
\href{https://www.libraw.org/docs/API-datastruct-eng.html#datastream_data:~:text=Per\%2Dchannel\%20black\%20level\%20correction}{\code{black_level_per_channel}}
metadata obtained with \code{rawpy}.}
}
\value{
Numeric \link[terra:SpatRaster-class]{terra::SpatRaster}:
\itemize{
\item single-layer if \code{only_blue = TRUE}.
\item multi-layer if \code{only_blue = FALSE}, with one layer per color per CFA
color (e.g., R, G, B).
}
Layers are named according to metadata in the raw file.
}
\description{
Read unprocessed sensor data from a camera RAW file and split the signal by
spectral band according to the in-camera color filter array (CFA). Use this
to obtain images with precise radiometry.
}
\details{
Uses Python \href{https://pypi.org/project/rawpy/}{\code{rawpy}} through \code{reticulate}
to access sensor data and black-level metadata. Optionally extracts only the
blue/cyan band.
}
\section{Check Python Accessibility}{


To ensure that R can access a Python installation, run the following test:

\if{html}{\out{<div class="sourceCode">}}\preformatted{reticulate::py_eval("1+1")

}\if{html}{\out{</div>}}

If R can access Python successfully, you will see \code{2} in the console. If not,
you will receive instructions on how to install Python.
}

\section{Create a Virtual Environment}{


After passing the Python accessibility test, create a virtual environment
using the following command:

\if{html}{\out{<div class="sourceCode">}}\preformatted{reticulate::virtualenv_create()

}\if{html}{\out{</div>}}
}

\section{Install \code{rawpy}}{


Install the rawpy package within the virtual environment:

\if{html}{\out{<div class="sourceCode">}}\preformatted{reticulate::py_install("rawpy")

}\if{html}{\out{</div>}}
}

\section{For RStudio Users}{


If you are an RStudio user who works with projects, you will need a
\emph{.Renviron} file in the root of each project. To create a \emph{.Renviron} file,
follow these steps:
\itemize{
\item Create a "New Blank File" named ".Renviron" (without an extension) in the
project's root directory.
\item Run bellow code:
}

\if{html}{\out{<div class="sourceCode">}}\preformatted{path <- file.path(reticulate::virtualenv_root(),
reticulate::virtualenv_list(), "Scripts", "python.exe")
paste("RETICULATE_PYTHON =", path)

}\if{html}{\out{</div>}}
\itemize{
\item Copy/paste the line from the console (the string between the quotes) into
the .Renviron file. This is an example \verb{RETICULATE_PYTHON = ~/.virtualenvs/r-reticulate/Scripts/python.exe}
\item Do not forget to save the changes
}

By following these steps, users can easily set up their environment to access
raw data efficiently, but it is not the only way of doing it, you might know
an easier or better one.

See the help page of \code{\link[=read_caim]{read_caim()}} and \code{\link[=fisheye_to_equidistant]{fisheye_to_equidistant()}} as a
complement to this help page. Further details about raw files can be found in
\insertCite{Diaz2024;textual}{rcaiman}.
}

\examples{
\dontrun{
file_name <- tempfile(fileext = ".NEF")
download.file("https://osf.io/s49py/download", file_name, mode = "wb")

# Geometric and radiometric corrections -----------------------------------
zenith_colrow <- c(1290, 988)/2
diameter <- 756
z <- zenith_image(diameter, lens("Nikon_FCE9"))
a <- azimuth_image(z)
m <- !is.na(z)
caim <- read_caim_raw(file_name, only_blue = TRUE)
caim <- crop_caim(caim, zenith_colrow - diameter/2, diameter, diameter)
caim <- correct_vignetting(caim, z, c(0.0638, -0.101))
caim <- fisheye_to_equidistant(caim, z, a, m, radius = 300,
                               k = 1, p = 1, rmax = 100)
}
}
\references{
\insertAllCited{}
}
\seealso{
\code{\link[=read_caim]{read_caim()}}
}
