#' resizableSplitUI
#'
#' @param id Module identifier
#' @param sidebarpane Content for page sidebar
#' @param maincontenttop Content for page main top pane 
#' @param maincontentbottom Content for page main bottom pane 
#' @param sidebar.percentage Width for sidebar as percentage
#' @param maincontenttop.percentage Height for page main top pane as percentage
#' @param return.only.taglist logical to return taglist instead of a bootstrapPage
#'
#' @return A Shiny UI object. If \code{return.only.taglist = TRUE}, returns a
#'   \code{shiny.tag.list} object containing the resizable split layout elements.
#'   If \code{return.only.taglist = FALSE} (default), returns a Shiny bootstrap
#'   page (from \code{shiny::bootstrapPage}) containing the resizable layout.
#'
#' @examples
#' if (interactive()) {
#' 
#'   ui <- resizableSplitUI("splitModule",
#'     sidebarpane = shiny::div("This is the sidebar"),
#'     maincontenttop = shiny::div("This is the maincontenttop"),
#'     maincontentbottom = shiny::div("This is the maincontentbottom")
#'   )
#'
#'   server <- function(input, output, session) { }
#'
#'   shinyApp(ui, server)
#' }
#'
#' @importFrom shiny bootstrapPage tagList tags div 
#' @importFrom shinyjs useShinyjs extendShinyjs
#' @export

resizableSplitUI <- function(id, sidebarpane, maincontenttop, maincontentbottom, sidebar.percentage=15, maincontenttop.percentage=85, return.only.taglist=FALSE) {
  
  ns <- shiny::NS(id)
  
  shiny::addResourcePath("resizableSplitLayout", system.file("www/resizableSplitLayout", package = "resizableSplitLayout"))
  
  jscode <- sprintf("
shinyjs.init = function() {

var horizontalSplit = Split(['#%s', '#%s'], {
  sizes: [%d, %d],
  gutterSize: 8,
  cursor: 'col-resize',
  onDragEnd: function() {
    console.log('Pane resized (horizontal)');
    refreshContent();
  }
});

var verticalSplit = Split(['#%s', '#%s'], {
  sizes: [%d, %d],
  direction: 'vertical',
  gutterSize: 8,
  cursor: 'row-resize',
  onDragEnd: function() {
    console.log('Pane resized (vertical)');
    refreshContent();
  }
});

// Function to trigger refresh of content
  function refreshContent() {
    console.log('Pane resized');
    
    // Not necessary:
    // Send a message to Shiny that content dimensions might have changed
    // Shiny.onInputChange('split_resized', new Date().getTime());
    
    // Explicitly trigger a window resize event
    const resizeEvent = new Event('resize');
    window.dispatchEvent(resizeEvent);
  }
}", ns("sidebar"), ns("maincontent"), sidebar.percentage, (100-sidebar.percentage), ns("maincontenttop"), ns("maincontentbottom"), maincontenttop.percentage, (100-maincontenttop.percentage))
  
	tags <- shiny::tagList(
	  shinyjs::useShinyjs(),
	  shinyjs::extendShinyjs(text = jscode, functions = c("shinyjs.init")),

	  shiny::tags$head(
	    shiny::includeCSS(system.file("www/resizableSplitLayout/resizableSplitLayout.css", package = "resizableSplitLayout")),
	    shiny::includeScript(system.file("www/resizableSplitLayout/split.min.js", package = "resizableSplitLayout"))
	    # shiny::tags$link(rel = "stylesheet", type = "text/css", href = "resizableSplitLayout/resizableSplitLayout.css"),
	    # shiny::tags$script(src = "resizableSplitLayout/split.min.js") # v.1.6.5
		),

		shiny::div(id = ns("sidebar"), class = "split split-horizontal content sidebar",
		           shiny::div(id = ns("sidebarpane"), class = "sidebarpane", sidebarpane)
		),

		shiny::div(id = ns("maincontent"), class = "split split-horizontal maincontent",
		    shiny::div(id = ns("maincontenttop"), class = "split content maincontentpane", maincontenttop),
		    shiny::div(id = ns("maincontentbottom"), class = "split content maincontentpane", maincontentbottom)
		)
	)
  
	if(return.only.taglist) {
	  tags # return just taglist if desired
	} else {
	  shiny::bootstrapPage(tags) # otherwise wrap in bootstrapPage since resizableSplit works best with bootstrap (not with flexpages)
	}
}


# Server Module
resizableSplitServer <- function(input, output, session) {
  # Placeholder for any server-side calls
}

#' Run Example App for resizableSplitLayout
#'
#' This function launches a simple Shiny application demonstrating how to use
#' the `resizableSplitLayout` module.
#'
#' @return No return value, called for side effects (launches a Shiny app)
#' @importFrom shiny shinyApp div observeEvent renderPlot
#' @export
#' @examples
#' # Run this function to see the module in action
#' if (interactive()) {
#'   resizableSplitDemo()
#' }

resizableSplitDemo <- function() {
  
  ui <- resizableSplitLayout::resizableSplitUI("resizableSplit",
                                               sidebarpane = shiny::div("This is the sidebarpane"),
                                               maincontenttop = shiny::div(height = "100%", style="height:100%;",
                                                 shiny::plotOutput("dataPlotter", height="100%", width="100%")
                                                  ),
                                               maincontentbottom = shiny::div("This is the maincontentbottom"),
                                               maincontenttop.percentage = 90
  )
  
  server <- function(input, output, session) {
    output$dataPlotter <- shiny::renderPlot({
      plot(Sepal.Length ~ Sepal.Width, datasets::iris)
    })
    
    shiny::observeEvent(input$split_resized, {
      cat("Split resized\n")
    })
  }
  
  shiny::shinyApp(ui, server)
}
